<?php
// Start output buffering
ob_start();

// koneksi dan session
include 'config.php';
include 'session.php';

error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once __DIR__ . '/dompdf/autoload.inc.php';
use Dompdf\Dompdf;

// Fungsi untuk mengamankan input
function sanitize_input($data) {
    return htmlspecialchars(stripslashes(trim($data)));
}

// Fungsi untuk format rupiah
function format_rupiah($amount) {
    return 'Rp ' . number_format($amount, 0, ',', '.');
}

// Class untuk menangani operasi kas kecil
class KasKecil {
    private $conn;
    
    public function __construct($connection) {
        $this->conn = $connection;
    }
    
    // Tambah transaksi baru
    public function tambahTransaksi($tanggal, $keterangan, $jenis, $jumlah) {
        $stmt = $this->conn->prepare("INSERT INTO kas_kecil (tanggal, keterangan, jenis, jumlah) VALUES (?, ?, ?, ?)");
        return $stmt->bind_param("sssd", $tanggal, $keterangan, $jenis, $jumlah) && $stmt->execute();
    }
    
    // Edit transaksi
    public function editTransaksi($id, $tanggal, $keterangan, $jumlah, $jenis) {
        $stmt = $this->conn->prepare("UPDATE kas_kecil SET tanggal=?, keterangan=?, jumlah=?, jenis=? WHERE id=?");
        return $stmt->bind_param("ssdsi", $tanggal, $keterangan, $jumlah, $jenis, $id) && $stmt->execute();
    }
    
    // Hapus transaksi
    public function hapusTransaksi($id) {
        $stmt = $this->conn->prepare("DELETE FROM kas_kecil WHERE id = ?");
        return $stmt->bind_param("i", $id) && $stmt->execute();
    }
    
    // Ambil semua transaksi
    public function getTransaksi($order = "tanggal DESC") {
        $result = $this->conn->query("SELECT * FROM kas_kecil ORDER BY $order");
        return $result->fetch_all(MYSQLI_ASSOC);
    }
    
    // Hitung saldo
    public function hitungSaldo() {
        $masuk = $this->conn->query("SELECT COALESCE(SUM(jumlah), 0) FROM kas_kecil WHERE jenis='masuk'")->fetch_row()[0];
        $keluar = $this->conn->query("SELECT COALESCE(SUM(jumlah), 0) FROM kas_kecil WHERE jenis='keluar'")->fetch_row()[0];
        
        return [
            'masuk' => $masuk,
            'keluar' => $keluar,
            'saldo' => $masuk - $keluar
        ];
    }
}

// Inisialisasi class
$kasKecil = new KasKecil($conn);
$message = '';
$messageType = '';

// Proses form submissions - HARUS DI ATAS SEBELUM ADA OUTPUT HTML
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    try {
        // Edit transaksi
        if (isset($_POST['edit_id'])) {
            $id = (int)$_POST['edit_id'];
            $tanggal = sanitize_input($_POST['tanggal']);
            $keterangan = sanitize_input($_POST['keterangan']);
            $jumlah = (float)$_POST['jumlah'];
            $jenis = sanitize_input($_POST['jenis']);
            
            if ($kasKecil->editTransaksi($id, $tanggal, $keterangan, $jumlah, $jenis)) {
                $message = "Transaksi berhasil diperbarui!";
                $messageType = "success";
            } else {
                $message = "Gagal memperbarui transaksi!";
                $messageType = "danger";
            }
        }
        // Hapus transaksi
        elseif (isset($_POST['hapus_id'])) {
            $hapus_id = (int)$_POST['hapus_id'];
            
            if ($kasKecil->hapusTransaksi($hapus_id)) {
                $message = "Transaksi berhasil dihapus!";
                $messageType = "success";
            } else {
                $message = "Gagal menghapus transaksi!";
                $messageType = "danger";
            }
        }
        // Tambah transaksi baru
        else {
            $tanggal = sanitize_input($_POST['tanggal']);
            $keterangan = sanitize_input($_POST['keterangan']);
            $jumlah = (float)$_POST['jumlah'];
            $jenis = sanitize_input($_POST['jenis']) ?? 'keluar';
            
            if ($kasKecil->tambahTransaksi($tanggal, $keterangan, $jenis, $jumlah)) {
                $message = "Transaksi berhasil ditambahkan!";
                $messageType = "success";
            } else {
                $message = "Gagal menambahkan transaksi!";
                $messageType = "danger";
            }
        }
        
        // Redirect untuk menghindari resubmission
        header("Location: " . $_SERVER['PHP_SELF'] . "?msg=" . urlencode($message) . "&type=" . $messageType);
        exit;
        
    } catch (Exception $e) {
        $message = "Terjadi kesalahan: " . $e->getMessage();
        $messageType = "danger";
    }
}

// Ambil data
$transaksi_tampil = $kasKecil->getTransaksi("tanggal DESC, id DESC");
$transaksi_hitung = $kasKecil->getTransaksi("tanggal ASC, id ASC");
$saldo_data = $kasKecil->hitungSaldo();

// Tampilkan pesan jika ada
if (isset($_GET['msg'])) {
    $message = $_GET['msg'];
    $messageType = $_GET['type'] ?? 'info';
}

// Include navbar setelah semua proses POST selesai
include 'navbar.php';
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kas Kecil - Manajemen Keuangan</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">

    <style>
        .saldo-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 15px;
        }
        .stats-card {
            border-radius: 10px;
            transition: transform 0.2s;
        }
        .stats-card:hover {
            transform: translateY(-5px);
        }
        .btn-action {
            margin: 2px;
        }
        .table-responsive {
            border-radius: 10px;
            overflow: hidden;
        }
    </style>
</head>
<body class="bg-light">
    <div class="container py-4">
        <!-- Header -->
        <div class="row mb-4">
            <div class="col">
                <h2 class="mb-0">Kas Kecil</h2>
                <p class="text-muted">Manajemen keuangan kas kecil</p>
            </div>
        </div>

        <!-- Alert Messages -->
        <?php if ($message): ?>
        <div class="alert alert-<?= $messageType ?> alert-dismissible fade show" role="alert">
            <?= $message ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>

        <!-- Stats Cards -->
        <div class="row mb-4">
            <div class="col-md-3 mb-3">
                <div class="card saldo-card stats-card">
                    <div class="card-body text-center">
                        <h6 class="card-title">Saldo Saat Ini</h6>
                        <h4><?= format_rupiah($saldo_data['saldo']) ?></h4>
                    </div>
                </div>
            </div>
            <div class="col-md-3 mb-3">
                <div class="card bg-success text-white stats-card">
                    <div class="card-body text-center">
                        <h6 class="card-title">Total Masuk</h6>
                        <h5><?= format_rupiah($saldo_data['masuk']) ?></h5>
                    </div>
                </div>
            </div>
            <div class="col-md-3 mb-3">
                <div class="card bg-danger text-white stats-card">
                    <div class="card-body text-center">
                        <h6 class="card-title">Total Keluar</h6>
                        <h5><?= format_rupiah($saldo_data['keluar']) ?></h5>
                    </div>
                </div>
            </div>
            <div class="col-md-3 mb-3">
                <div class="card bg-info text-white stats-card">
                    <div class="card-body text-center">
                        <h6 class="card-title">Total Transaksi</h6>
                        <h5><?= count($transaksi_tampil) ?></h5>
                    </div>
                </div>
            </div>
        </div>

        <!-- Action Buttons -->
        <div class="row mb-4">
            <div class="col-md-8">
                <button class="btn btn-success btn-lg me-2" data-bs-toggle="modal" data-bs-target="#modalMasuk">
                    + Kas Masuk
                </button>
                <button class="btn btn-primary btn-lg" data-bs-toggle="modal" data-bs-target="#modalKeluar">
                    + Kas Keluar
                </button>
            </div>
        </div>

        <!-- Print Form -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0">Cetak Laporan</h5>
            </div>
            <div class="card-body">
                <form method="GET" action="cetak_kas.php" target="_blank" class="row g-3 align-items-end">
                    <div class="col-md-4">
                        <label for="tanggal_awal" class="form-label">Dari Tanggal</label>
                        <input type="date" class="form-control" name="tanggal_awal" required value="<?= date('Y-m-d') ?>">
                    </div>
                    <div class="col-md-4">
                        <label for="tanggal_akhir" class="form-label">Sampai Tanggal</label>
                        <input type="date" class="form-control" name="tanggal_akhir" required value="<?= date('Y-m-d') ?>">
                    </div>
                    <div class="col-md-4">
                        <button type="submit" class="btn btn-danger w-100">
                            🖨 Cetak PDF
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Tabel Transaksi -->
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Daftar Transaksi</h5>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-striped table-hover mb-0">
                        <thead class="table-dark">
                            <tr>
                                <th>Tanggal</th>
                                <th>Keterangan</th>
                                <th>Jenis</th>
                                <th>Jumlah</th>
                                <th>Saldo Berjalan</th>
                                <th width="150">Aksi</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($transaksi_hitung)): ?>
                            <tr>
                                <td colspan="6" class="text-center py-4">
                                    <p class="text-muted">Belum ada transaksi</p>
                                </td>
                            </tr>
                            <?php else: ?>
                            <?php
                            $saldo_berjalan = 0;
                            // Tampilkan dari yang terbaru tapi hitung saldo dari yang terlama
                            foreach ($transaksi_tampil as $row):
                                // Hitung ulang saldo berjalan untuk setiap baris
                                $saldo_temp = 0;
                                foreach ($transaksi_hitung as $temp_row) {
                                    if ($temp_row['tanggal'] < $row['tanggal'] || 
                                        ($temp_row['tanggal'] == $row['tanggal'] && $temp_row['id'] <= $row['id'])) {
                                        if ($temp_row['jenis'] == 'masuk') {
                                            $saldo_temp += $temp_row['jumlah'];
                                        } else {
                                            $saldo_temp -= $temp_row['jumlah'];
                                        }
                                    }
                                }
                                $saldo_berjalan = $saldo_temp;
                            ?>
                            <tr>
                                <td>
                                    <?= date('d/m/Y', strtotime($row['tanggal'])) ?>
                                </td>
                                <td><?= htmlspecialchars($row['keterangan']) ?></td>
                                <td>
                                    <?php if ($row['jenis'] === 'masuk'): ?>
                                        <span class="badge bg-success">
                                            Masuk
                                        </span>
                                    <?php else: ?>
                                        <span class="badge bg-danger">
                                            Keluar
                                        </span>
                                    <?php endif; ?>
                                </td>
                                <td class="<?= $row['jenis'] === 'masuk' ? 'text-success' : 'text-danger' ?> fw-bold">
                                    <?= format_rupiah($row['jumlah']) ?>
                                </td>
                                <td class="fw-bold"><?= format_rupiah($saldo_berjalan) ?></td>
                                <td>
                                    <div class="btn-group btn-group-sm" role="group">
                                        <button class="btn btn-outline-warning btn-action" 
                                                data-bs-toggle="modal" 
                                                data-bs-target="#modalEdit<?= $row['id'] ?>"
                                                title="Edit">
                                            Edit
                                        </button>
                                        <button class="btn btn-outline-danger btn-action" 
                                                onclick="confirmDelete(<?= $row['id'] ?>)"
                                                title="Hapus">
                                            Hapus
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal Kas Masuk -->
    <div class="modal fade" id="modalMasuk" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <form method="post" class="modal-content">
                <div class="modal-header bg-success text-white">
                    <h5 class="modal-title">
                        Input Kas Masuk
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">
                            Tanggal
                        </label>
                        <input type="date" name="tanggal" class="form-control" required value="<?= date('Y-m-d') ?>">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">
                            Keterangan
                        </label>
                        <input type="text" name="keterangan" class="form-control" 
                               placeholder="Masukkan keterangan..." required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">
                            Jumlah (Rp)
                        </label>
                        <input type="number" step="0.01" name="jumlah" class="form-control" 
                               placeholder="0" required min="0">
                    </div>
                    <input type="hidden" name="jenis" value="masuk">
                </div>
                <div class="modal-footer">
                    <button type="submit" class="btn btn-success">
                        Simpan
                    </button>
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        Batal
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Modal Kas Keluar -->
    <div class="modal fade" id="modalKeluar" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <form method="post" class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title">
                        Input Kas Keluar
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">
                            Tanggal
                        </label>
                        <input type="date" name="tanggal" class="form-control" required value="<?= date('Y-m-d') ?>">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">
                            Keterangan
                        </label>
                        <input type="text" name="keterangan" class="form-control" 
                               placeholder="Masukkan keterangan..." required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">
                            Jumlah (Rp)
                        </label>
                        <input type="number" step="0.01" name="jumlah" class="form-control" 
                               placeholder="0" required min="0">
                    </div>
                    <input type="hidden" name="jenis" value="keluar">
                </div>
                <div class="modal-footer">
                    <button type="submit" class="btn btn-primary">
                        Simpan
                    </button>
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        Batal
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Modal Edit untuk setiap transaksi -->
    <?php foreach ($transaksi_tampil as $row): ?>
    <div class="modal fade" id="modalEdit<?= $row['id'] ?>" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <form method="post" class="modal-content">
                <div class="modal-header bg-warning text-dark">
                    <h5 class="modal-title">
                        Edit Transaksi
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <input type="hidden" name="edit_id" value="<?= $row['id'] ?>">
                    <div class="mb-3">
                        <label class="form-label">
                            Tanggal
                        </label>
                        <input type="date" name="tanggal" class="form-control" 
                               value="<?= $row['tanggal'] ?>" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">
                            Keterangan
                        </label>
                        <input type="text" name="keterangan" class="form-control" 
                               value="<?= htmlspecialchars($row['keterangan']) ?>" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">
                            Jumlah (Rp)
                        </label>
                        <input type="number" step="0.01" name="jumlah" class="form-control" 
                               value="<?= $row['jumlah'] ?>" required min="0">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">
                            Jenis
                        </label>
                        <select name="jenis" class="form-select">
                            <option value="masuk" <?= $row['jenis'] == 'masuk' ? 'selected' : '' ?>>
                                Masuk
                            </option>
                            <option value="keluar" <?= $row['jenis'] == 'keluar' ? 'selected' : '' ?>>
                                Keluar
                            </option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="submit" class="btn btn-warning">
                        Update
                    </button>
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        Batal
                    </button>
                </div>
            </form>
        </div>
    </div>
    <?php endforeach; ?>

    <!-- Hidden Form untuk Delete -->
    <form id="deleteForm" method="post" style="display: none;">
        <input type="hidden" id="hapus_id" name="hapus_id">
    </form>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    
    <script>
        // Konfirmasi delete dengan SweetAlert
        function confirmDelete(id) {
            Swal.fire({
                title: 'Yakin ingin menghapus?',
                text: "Data yang dihapus tidak dapat dikembalikan!",
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#d33',
                cancelButtonColor: '#3085d6',
                confirmButtonText: 'Ya, Hapus!',
                cancelButtonText: 'Batal'
            }).then((result) => {
                if (result.isConfirmed) {
                    document.getElementById('hapus_id').value = id;
                    document.getElementById('deleteForm').submit();
                }
            });
        }

        // Auto dismiss alerts
        setTimeout(function() {
            var alerts = document.querySelectorAll('.alert');
            alerts.forEach(function(alert) {
                var bsAlert = new bootstrap.Alert(alert);
                bsAlert.close();
            });
        }, 5000);

        // Format number input
        document.querySelectorAll('input[type="number"]').forEach(function(input) {
            input.addEventListener('input', function() {
                if (this.value < 0) this.value = 0;
            });
        });
    </script>
</body>
</html>
<?php
// End output buffering
ob_end_flush();
?>