<?php
// Error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Include files
include 'config.php';
include 'session.php';
include 'navbar.php';

// Check database connection
if (!isset($conn) || !$conn) {
    die("Error: Koneksi database tidak tersedia");
}

// Parameters
$periode = isset($_GET['periode']) ? $_GET['periode'] : date('Y-m');
$customer_id = isset($_GET['customer_id']) ? (int)$_GET['customer_id'] : 0;
$export = isset($_GET['export']) ? $_GET['export'] : '';

// Functions
function formatCurrency($amount) {
    return 'Rp ' . number_format($amount, 0, ',', '.');
}

function formatTanggal($tanggal) {
    return date('d/m/Y', strtotime($tanggal));
}

// Build query
$where_conditions = ["1=1"];
if ($periode != 'all') {
    $where_conditions[] = "DATE_FORMAT(p.tanggal, '%Y-%m') = '$periode'";
}
if ($customer_id > 0) {
    $where_conditions[] = "p.customer_id = $customer_id";
}
$where_clause = "WHERE " . implode(" AND ", $where_conditions);

// PERBAIKAN: Main query - Detail penjualan dengan info pembelian
$query = "
    SELECT 
        p.id as penjualan_id,
        p.no_invoice,
        p.tanggal,
        p.jenis,
        p.grand_total,
        p.rekening,
        c.nama as customer_nama,
        c.kode as customer_kode,
        c.kota as customer_kota,
        pd.id as detail_id,
        pd.qty,
        pd.harga as harga_jual,
        (pd.qty * pd.harga) as total_jual,
        pd.supplier as supplier_id,
        pd.harga_beli,
        pd.nota,
        pi.nama as produk_nama,
        s.nama as supplier_nama,
        ((pd.qty * pd.harga) - (pd.qty * COALESCE(pd.harga_beli, 0))) as untung
    FROM penjualan p
    LEFT JOIN customer c ON p.customer_id = c.id
    LEFT JOIN penjualan_detail pd ON p.id = pd.penjualan_id
    LEFT JOIN produk_interest pi ON pd.produk_id = pi.id
    LEFT JOIN supplier s ON pd.supplier = s.id
    $where_clause
    ORDER BY p.tanggal DESC, p.id DESC, pd.id ASC
";

$result = mysqli_query($conn, $query);
if (!$result) {
    die("Query error: " . mysqli_error($conn));
}

// PERBAIKAN: Query untuk rekapan per invoice
$recap_query = "
    SELECT 
        p.id as penjualan_id,
        p.no_invoice,
        p.tanggal,
        p.jenis,
        p.grand_total,
        p.rekening,
        c.nama as customer_nama,
        c.kode as customer_kode,
        c.kota as customer_kota,
        COUNT(pd.id) as jumlah_item,
        SUM(pd.qty * pd.harga) as total_penjualan_detail,
        SUM(pd.qty * COALESCE(pd.harga_beli, 0)) as total_pembelian_detail,
        SUM((pd.qty * pd.harga) - (pd.qty * COALESCE(pd.harga_beli, 0))) as laba_invoice
    FROM penjualan p
    LEFT JOIN customer c ON p.customer_id = c.id
    LEFT JOIN penjualan_detail pd ON p.id = pd.penjualan_id
    $where_clause
    GROUP BY p.id, p.no_invoice, p.tanggal, p.jenis, p.grand_total, p.rekening, c.nama, c.kode, c.kota
    ORDER BY p.tanggal DESC, p.id DESC
";

$recap_result = mysqli_query($conn, $recap_query);
if (!$recap_result) {
    die("Recap query error: " . mysqli_error($conn));
}

// Handle Excel Export
if ($export == 'excel') {
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment; filename="laporan_detail_' . date('Y-m-d') . '.xls"');
    
    // Export Rekapan Invoice
    echo "<h3>REKAPAN PER INVOICE</h3>";
    echo "<table border='1'>";
    echo "<tr>";
    echo "<th>No Invoice</th>";
    echo "<th>Tanggal</th>";
    echo "<th>Customer</th>";
    echo "<th>Kota</th>";
    echo "<th>Items</th>";
    echo "<th>Total Penjualan</th>";
    echo "<th>Total Pembelian</th>";
    echo "<th>Laba Invoice</th>";
    echo "<th>Margin %</th>";
    echo "<th>Jenis</th>";
    echo "<th>Rekening</th>";
    echo "</tr>";
    
    mysqli_data_seek($recap_result, 0);
    while ($row = mysqli_fetch_assoc($recap_result)) {
        $margin_invoice = $row['total_penjualan_detail'] > 0 ? 
            (($row['laba_invoice'] / $row['total_penjualan_detail']) * 100) : 0;
        
        echo "<tr>";
        echo "<td>" . $row['no_invoice'] . "</td>";
        echo "<td>" . formatTanggal($row['tanggal']) . "</td>";
        echo "<td>[" . $row['customer_kode'] . "] " . $row['customer_nama'] . "</td>";
        echo "<td>" . $row['customer_kota'] . "</td>";
        echo "<td>" . $row['jumlah_item'] . "</td>";
        echo "<td>" . $row['total_penjualan_detail'] . "</td>";
        echo "<td>" . $row['total_pembelian_detail'] . "</td>";
        echo "<td>" . $row['laba_invoice'] . "</td>";
        echo "<td>" . number_format($margin_invoice, 1) . "%</td>";
        echo "<td>" . $row['jenis'] . "</td>";
        echo "<td>" . $row['rekening'] . "</td>";
        echo "</tr>";
    }
    echo "</table>";
    
    echo "<br><br>";
    
    // Export Detail
    echo "<h3>DETAIL PER ITEM</h3>";
    echo "<table border='1'>";
    echo "<tr>";
    echo "<th>No Invoice</th>";
    echo "<th>Tanggal</th>";
    echo "<th>Customer</th>";
    echo "<th>Kota</th>";
    echo "<th>Produk</th>";
    echo "<th>Qty</th>";
    echo "<th>Harga Jual</th>";
    echo "<th>Total Jual</th>";
    echo "<th>Supplier</th>";
    echo "<th>Harga Beli</th>";
    echo "<th>Total Beli</th>";
    echo "<th>Laba</th>";
    echo "<th>Margin %</th>";
    echo "<th>Nota</th>";
    echo "<th>Jenis</th>";
    echo "<th>Rekening</th>";
    echo "</tr>";
    
    mysqli_data_seek($result, 0);
    while ($row = mysqli_fetch_assoc($result)) {
        $total_beli = $row['qty'] * ($row['harga_beli'] ?? 0);
        $margin_persen = $row['harga_jual'] > 0 ? (($row['harga_jual'] - ($row['harga_beli'] ?? 0)) / $row['harga_jual'] * 100) : 0;
        
        echo "<tr>";
        echo "<td>" . $row['no_invoice'] . "</td>";
        echo "<td>" . formatTanggal($row['tanggal']) . "</td>";
        echo "<td>[" . $row['customer_kode'] . "] " . $row['customer_nama'] . "</td>";
        echo "<td>" . $row['customer_kota'] . "</td>";
        echo "<td>" . $row['produk_nama'] . "</td>";
        echo "<td>" . $row['qty'] . "</td>";
        echo "<td>" . $row['harga_jual'] . "</td>";
        echo "<td>" . $row['total_jual'] . "</td>";
        echo "<td>" . $row['supplier_nama'] . "</td>";
        echo "<td>" . ($row['harga_beli'] ?? 0) . "</td>";
        echo "<td>" . $total_beli . "</td>";
        echo "<td>" . $row['untung'] . "</td>";
        echo "<td>" . number_format($margin_persen, 1) . "%</td>";
        echo "<td>" . $row['nota'] . "</td>";
        echo "<td>" . $row['jenis'] . "</td>";
        echo "<td>" . $row['rekening'] . "</td>";
        echo "</tr>";
    }
    echo "</table>";
    exit;
}

// Generate periode options
$periode_options = array();
for ($i = 0; $i < 12; $i++) {
    $date = date('Y-m', strtotime("-$i month"));
    $label = date('F Y', strtotime("-$i month"));
    $periode_options[] = array('value' => $date, 'label' => $label);
}

// PERBAIKAN: Calculate summary - gunakan perhitungan yang konsisten
$total_penjualan = 0;
$total_pembelian = 0;
$total_untung = 0;
$jumlah_transaksi = 0;

if ($result) {
    mysqli_data_seek($result, 0);
    $prev_invoice = '';
    while ($row = mysqli_fetch_assoc($result)) {
        if ($row['no_invoice'] != $prev_invoice) {
            $jumlah_transaksi++;
            $prev_invoice = $row['no_invoice'];
        }
        // PERBAIKAN: Gunakan total_jual yang sudah dihitung ulang
        $total_penjualan += $row['total_jual'];
        $total_pembelian += ($row['qty'] * ($row['harga_beli'] ?? 0));
        $total_untung += $row['untung'];
    }
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Laporan Penjualan Detail - SBA System</title>
    
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 0;
            padding: 20px;
            background-color: #f5f5f5;
            font-size: 13px;
        }

        .container {
            max-width: 100%;
            background: white;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }

        .header {
            text-align: center;
            margin-bottom: 30px;
            border-bottom: 2px solid #333;
            padding-bottom: 15px;
        }

        .header h1 {
            margin: 0;
            color: #333;
            font-size: 24px;
        }

        .filter-section {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            border: 1px solid #ddd;
        }

        .filter-section h3 {
            margin: 0 0 15px 0;
            color: #333;
        }

        .form-group {
            display: inline-block;
            margin-right: 20px;
            margin-bottom: 10px;
        }

        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }

        .form-group select, .form-group input {
            padding: 8px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 13px;
        }

        .btn {
            padding: 8px 15px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 13px;
            text-decoration: none;
            display: inline-block;
            margin-right: 10px;
        }

        .btn-primary {
            background: #007bff;
            color: white;
        }

        .btn-success {
            background: #28a745;
            color: white;
        }

        .btn-info {
            background: #17a2b8;
            color: white;
        }

        .btn:hover {
            opacity: 0.8;
        }

        .summary-section {
            background: #e8f4fd;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            border: 1px solid #bee5eb;
        }

        .summary-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
        }

        .summary-item {
            text-align: center;
        }

        .summary-item h4 {
            margin: 0;
            color: #0c5460;
            font-size: 18px;
        }

        .summary-item p {
            margin: 5px 0 0 0;
            color: #666;
        }

        .table-container {
            overflow-x: auto;
            border: 1px solid #ddd;
            border-radius: 5px;
            margin-bottom: 30px;
        }

        .table {
            width: 100%;
            border-collapse: collapse;
            margin: 0;
            background: white;
        }

        .table th {
            background: #f8f9fa;
            color: #333;
            font-weight: bold;
            padding: 10px 8px;
            text-align: left;
            border-bottom: 2px solid #dee2e6;
            position: sticky;
            top: 0;
            font-size: 12px;
            white-space: nowrap;
        }

        .table td {
            padding: 8px;
            border-bottom: 1px solid #dee2e6;
            vertical-align: top;
            font-size: 12px;
        }

        .table tbody tr:hover {
            background: #f8f9fa;
        }

        .table tbody tr:nth-child(even) {
            background: #fafafa;
        }

        .amount {
            text-align: right;
            font-weight: bold;
            color: #28a745;
        }

        .loss {
            color: #dc3545;
        }

        .profit {
            color: #28a745;
        }

        .text-center {
            text-align: center;
        }

        .invoice-group {
            border-left: 3px solid #007bff;
            background: #f0f8ff;
        }

        .nota-link {
            color: #007bff;
            text-decoration: none;
            font-size: 11px;
        }

        .nota-link:hover {
            text-decoration: underline;
        }

        .status-badge {
            padding: 3px 8px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: bold;
        }

        .status-ppn {
            background: #d1ecf1;
            color: #0c5460;
        }

        .status-non-ppn {
            background: #fff3cd;
            color: #856404;
        }

        .section-header {
            background: #343a40;
            color: white;
            padding: 12px 15px;
            margin: 30px 0 15px 0;
            border-radius: 5px;
            font-weight: bold;
            font-size: 16px;
        }

        .recap-table {
            background: #fff8e1;
        }

        .recap-table th {
            background: #ffc107;
            color: #212529;
        }

        .recap-row {
            background: #fffbf0;
            font-weight: bold;
        }

        .tab-buttons {
            margin-bottom: 20px;
        }

        .tab-button {
            padding: 10px 20px;
            border: 1px solid #ddd;
            background: #f8f9fa;
            cursor: pointer;
            border-radius: 5px 5px 0 0;
            margin-right: 5px;
            display: inline-block;
        }

        .tab-button.active {
            background: #007bff;
            color: white;
            border-color: #007bff;
        }

        .tab-content {
            display: none;
        }

        .tab-content.active {
            display: block;
        }

        /* TAMBAHAN: Style untuk highlight perbaikan */
        .fix-highlight {
            background: #d4edda;
            border: 1px solid #c3e6cb;
            color: #155724;
            padding: 10px;
            margin: 10px 0;
            border-radius: 5px;
            font-size: 12px;
        }

        @media print {
            .filter-section, .btn, .tab-buttons, .fix-highlight {
                display: none;
            }
            
            body {
                background: white;
                font-size: 11px;
            }
            
            .table th, .table td {
                padding: 5px;
                font-size: 10px;
            }

            .tab-content {
                display: block !important;
            }
        }

        @media (max-width: 768px) {
            .form-group {
                display: block;
                margin-right: 0;
                margin-bottom: 15px;
            }
            
            .summary-grid {
                grid-template-columns: 1fr;
            }
            
            .table {
                font-size: 11px;
            }
            
            .table th, .table td {
                padding: 6px 4px;
            }
        }
    </style>
</head>
<body>

<div class="container">
    <div class="header">
        <h1>📊 LAPORAN PENJUALAN DETAIL</h1>
        <p>Sistem Manajemen Bisnis SBA</p>
    </div>



    <!-- Filter Section -->
    <div class="filter-section">
        <h3>🔍 Filter Laporan</h3>
        <form method="GET" action="">
            <div class="form-group">
                <label>Periode:</label>
                <select name="periode">
                    <option value="all" <?php echo $periode == 'all' ? 'selected' : ''; ?>>Semua Periode</option>
                    <?php foreach ($periode_options as $option): ?>
                        <option value="<?php echo $option['value']; ?>" 
                                <?php echo $periode == $option['value'] ? 'selected' : ''; ?>>
                            <?php echo $option['label']; ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="form-group">
                <label>Customer:</label>
                <select name="customer_id">
                    <option value="0">Semua Customer</option>
                    <?php
                    $customer_query = mysqli_query($conn, "SELECT id, nama, kode FROM customer WHERE aktif = 1 ORDER BY nama");
                    if ($customer_query) {
                        while ($customer = mysqli_fetch_assoc($customer_query)): ?>
                            <option value="<?php echo $customer['id']; ?>" 
                                    <?php echo $customer_id == $customer['id'] ? 'selected' : ''; ?>>
                                [<?php echo $customer['kode']; ?>] <?php echo $customer['nama']; ?>
                            </option>
                        <?php endwhile;
                    }
                    ?>
                </select>
            </div>
            
            <div class="form-group">
                <button type="submit" class="btn btn-primary">📋 Filter</button>
                <a href="?export=excel&<?php echo http_build_query($_GET); ?>" 
                   class="btn btn-success">📊 Export Excel</a>
                <button type="button" onclick="window.print()" class="btn btn-success">🖨️ Print</button>
            </div>
        </form>
    </div>

    <!-- Summary Section -->
    <div class="summary-section">
        <div class="summary-grid">
            <div class="summary-item">
                <h4><?php echo formatCurrency($total_penjualan); ?></h4>
                <p>Total Penjualan</p>
            </div>
            <div class="summary-item">
                <h4><?php echo formatCurrency($total_pembelian); ?></h4>
                <p>Total Pembelian</p>
            </div>
            <div class="summary-item">
                <h4 class="<?php echo $total_untung >= 0 ? 'profit' : 'loss'; ?>">
                    <?php echo formatCurrency($total_untung); ?>
                </h4>
                <p>Total Laba</p>
            </div>
            <div class="summary-item">
                <h4><?php echo number_format($jumlah_transaksi); ?></h4>
                <p>Jumlah Transaksi</p>
            </div>
            <div class="summary-item">
                <h4><?php echo $total_penjualan > 0 ? number_format(($total_untung / $total_penjualan) * 100, 1) : 0; ?>%</h4>
                <p>Margin Rata-rata</p>
            </div>
        </div>
    </div>

    <!-- Tab Buttons -->
    <div class="tab-buttons">
        <button class="tab-button active" onclick="showTab('recap')">📊 Rekapan Per Invoice</button>
        <button class="tab-button" onclick="showTab('detail')">📋 Detail Per Item</button>
    </div>

    <!-- Rekapan Per Invoice Tab -->
    <div id="recap" class="tab-content active">
        <div class="section-header">
            📊 REKAPAN LABA PER INVOICE
        </div>
        
        <div class="table-container">
            <table class="table recap-table">
                <thead>
                    <tr>
                        <th>No Invoice</th>
                        <th>Tanggal</th>
                        <th>Customer</th>
                        <th>Kota</th>
                        <th>Items</th>
                        <th>Total Penjualan</th>
                        <th>Total Pembelian</th>
                        <th>Laba Invoice</th>
                        <th>Margin %</th>
                        <th>Jenis</th>
                        <th>Rekening</th>
                    </tr>
                </thead>
                <tbody>
                    <?php 
                    if ($recap_result && mysqli_num_rows($recap_result) > 0): 
                        mysqli_data_seek($recap_result, 0);
                        while ($row = mysqli_fetch_assoc($recap_result)): 
                            $margin_invoice = $row['total_penjualan_detail'] > 0 ? 
                                (($row['laba_invoice'] / $row['total_penjualan_detail']) * 100) : 0;
                            ?>
                            <tr class="recap-row">
                                <td><strong><?php echo htmlspecialchars($row['no_invoice']); ?></strong></td>
                                <td><?php echo formatTanggal($row['tanggal']); ?></td>
                                <td>
                                    <strong>[<?php echo htmlspecialchars($row['customer_kode']); ?>]</strong><br>
                                    <?php echo htmlspecialchars($row['customer_nama']); ?>
                                </td>
                                <td><?php echo htmlspecialchars($row['customer_kota']); ?></td>
                                <td class="text-center"><?php echo number_format($row['jumlah_item']); ?></td>
                                <td class="amount"><?php echo formatCurrency($row['total_penjualan_detail']); ?></td>
                                <td class="amount"><?php echo formatCurrency($row['total_pembelian_detail']); ?></td>
                                <td class="amount <?php echo $row['laba_invoice'] >= 0 ? 'profit' : 'loss'; ?>">
                                    <?php echo formatCurrency($row['laba_invoice']); ?>
                                </td>
                                <td class="text-center <?php echo $margin_invoice >= 0 ? 'profit' : 'loss'; ?>">
                                    <?php echo number_format($margin_invoice, 1); ?>%
                                </td>
                                <td>
                                    <span class="status-badge <?php echo $row['jenis'] == 'PPN' ? 'status-ppn' : 'status-non-ppn'; ?>">
                                        <?php echo htmlspecialchars($row['jenis']); ?>
                                    </span>
                                </td>
                                <td><?php echo htmlspecialchars($row['rekening']); ?></td>
                            </tr>
                        <?php endwhile; 
                    else: ?>
                        <tr>
                            <td colspan="11" class="text-center">Tidak ada data untuk periode yang dipilih</td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Detail Per Item Tab -->
    <div id="detail" class="tab-content">
        <div class="section-header">
            📋 DETAIL LABA PER ITEM
        </div>
        
        <div class="table-container">
            <table class="table">
                <thead>
                    <tr>
                        <th>No Invoice</th>
                        <th>Tanggal</th>
                        <th>Customer</th>
                        <th>Kota</th>
                        <th>Produk</th>
                        <th>Qty</th>
                        <th>Harga Jual</th>
                        <th>Total Jual</th>
                        <th>Supplier</th>
                        <th>Harga Beli</th>
                        <th>Total Beli</th>
                        <th>Laba</th>
                        <th>Margin %</th>
                        <th>Nota</th>
                        <th>Jenis</th>
                        <th>Rekening</th>
                    </tr>
                </thead>
                <tbody>
                    <?php 
                    if ($result && mysqli_num_rows($result) > 0): 
                        mysqli_data_seek($result, 0);
                        $current_invoice = '';
                        while ($row = mysqli_fetch_assoc($result)): 
                            $total_beli = $row['qty'] * ($row['harga_beli'] ?? 0);
                            $margin_persen = $row['harga_jual'] > 0 ? (($row['harga_jual'] - ($row['harga_beli'] ?? 0)) / $row['harga_jual'] * 100) : 0;
                            $is_new_invoice = $row['no_invoice'] != $current_invoice;
                            $current_invoice = $row['no_invoice'];
                            ?>
                            <tr <?php echo $is_new_invoice ? 'class="invoice-group"' : ''; ?>>
                                <td>
                                    <?php if ($is_new_invoice): ?>
                                        <strong><?php echo htmlspecialchars($row['no_invoice']); ?></strong>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($is_new_invoice): ?>
                                        <?php echo formatTanggal($row['tanggal']); ?>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($is_new_invoice): ?>
                                        <strong>[<?php echo htmlspecialchars($row['customer_kode']); ?>]</strong><br>
                                        <?php echo htmlspecialchars($row['customer_nama']); ?>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($is_new_invoice): ?>
                                        <?php echo htmlspecialchars($row['customer_kota']); ?>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo htmlspecialchars($row['produk_nama']); ?></td>
                                <td class="text-center"><?php echo number_format($row['qty']); ?></td>
                                <td class="amount"><?php echo formatCurrency($row['harga_jual']); ?></td>
                                <td class="amount"><?php echo formatCurrency($row['total_jual']); ?></td>
                                <td><?php echo htmlspecialchars($row['supplier_nama'] ?? '-'); ?></td>
                                <td class="amount"><?php echo formatCurrency($row['harga_beli'] ?? 0); ?></td>
                                <td class="amount"><?php echo formatCurrency($total_beli); ?></td>
                                <td class="amount <?php echo $row['untung'] >= 0 ? 'profit' : 'loss'; ?>">
                                    <?php echo formatCurrency($row['untung']); ?>
                                </td>
                                <td class="text-center <?php echo $margin_persen >= 0 ? 'profit' : 'loss'; ?>">
                                    <?php echo number_format($margin_persen, 1); ?>%
                                </td>
                                <td class="text-center">
                                    <?php if ($row['nota']): ?>
                                        <a href="nota_uploads/<?php echo htmlspecialchars($row['nota']); ?>" 
                                           target="_blank" class="nota-link">
                                            📄 Lihat
                                        </a>
                                    <?php else: ?>
                                        -
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($is_new_invoice): ?>
                                        <span class="status-badge <?php echo $row['jenis'] == 'PPN' ? 'status-ppn' : 'status-non-ppn'; ?>">
                                            <?php echo htmlspecialchars($row['jenis']); ?>
                                        </span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($is_new_invoice): ?>
                                        <?php echo htmlspecialchars($row['rekening']); ?>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endwhile; 
                    else: ?>
                        <tr>
                            <td colspan="16" class="text-center">Tidak ada data untuk periode yang dipilih</td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <div style="margin-top: 20px; text-align: center; color: #666; font-size: 12px;">
        <p>Laporan dibuat pada: <?php echo date('d F Y H:i:s'); ?></p>
        <p>Total Record: <?php echo $result ? mysqli_num_rows($result) : 0; ?> item | 
           Total Invoice: <?php echo $recap_result ? mysqli_num_rows($recap_result) : 0; ?> invoice</p>
        
    </div>
</div>

<script>
function showTab(tabName) {
    // Hide all tab contents
    var tabContents = document.querySelectorAll('.tab-content');
    for (var i = 0; i < tabContents.length; i++) {
        tabContents[i].classList.remove('active');
    }
    
    // Remove active class from all tab buttons
    var tabButtons = document.querySelectorAll('.tab-button');
    for (var i = 0; i < tabButtons.length; i++) {
        tabButtons[i].classList.remove('active');
    }
    
    // Show selected tab content
    document.getElementById(tabName).classList.add('active');
    
    // Add active class to clicked button
    event.target.classList.add('active');
}

// Auto switch to recap tab on load
document.addEventListener('DOMContentLoaded', function() {
    showTab('recap');
});
</script>

</body>
</html>