<?php
include 'config.php';

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Start session if not started
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Check if user is logged in
if (empty($_SESSION['username'])) {
    header("Location: login.php");
    exit();
}

// Check if connection exists
if (!isset($conn) || !$conn) {
    die('Error: Database connection tidak tersedia!');
}

$message = '';
$error = '';

// Get current user ID from username
$current_user_id = null;
if (isset($_SESSION['username'])) {
    $username = mysqli_real_escape_string($conn, $_SESSION['username']);
    $user_query = mysqli_query($conn, "SELECT user_id FROM user WHERE user_username = '$username'");
    if ($user_query && mysqli_num_rows($user_query) > 0) {
        $user_row = mysqli_fetch_assoc($user_query);
        $current_user_id = (int)$user_row['user_id'];
    }
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Check if we have current user ID
    if (!$current_user_id) {
        $error = "Error: Tidak dapat menentukan user ID dari username: " . $_SESSION['username'];
    } else {
        // Start transaction
        mysqli_autocommit($conn, false);
        
        try {
            // Validate required fields
            if (empty($_POST['tanggal']) || empty($_POST['tempat'])) {
                throw new Exception('Tanggal dan tempat meeting harus diisi');
            }
            
            // Escape input data
            $tanggal = mysqli_real_escape_string($conn, $_POST['tanggal']);
            $tempat = mysqli_real_escape_string($conn, $_POST['tempat']);
            $jenis_meeting = mysqli_real_escape_string($conn, $_POST['jenis_meeting']);
            $waktu_mulai = !empty($_POST['waktu_mulai']) ? "'" . mysqli_real_escape_string($conn, $_POST['waktu_mulai']) . "'" : 'NULL';
            $waktu_selesai = !empty($_POST['waktu_selesai']) ? "'" . mysqli_real_escape_string($conn, $_POST['waktu_selesai']) . "'" : 'NULL';
            $customer_id = !empty($_POST['customer_id']) ? (int)$_POST['customer_id'] : 'NULL';
            $supplier_id = !empty($_POST['supplier_id']) ? (int)$_POST['supplier_id'] : 'NULL';
            
            // Insert meeting data
            $meeting_sql = "
                INSERT INTO meeting (tanggal, waktu_mulai, waktu_selesai, tempat, jenis_meeting, 
                                   customer_id, supplier_id, created_by, kode_meeting) 
                VALUES ('$tanggal', $waktu_mulai, $waktu_selesai, '$tempat', '$jenis_meeting', 
                        $customer_id, $supplier_id, $current_user_id, '')
            ";
            
            if (!mysqli_query($conn, $meeting_sql)) {
                throw new Exception('Error inserting meeting: ' . mysqli_error($conn));
            }
            
            $meeting_id = mysqli_insert_id($conn);
            
            // Insert agenda items if any
            if (!empty($_POST['agenda_subject']) && is_array($_POST['agenda_subject'])) {
                foreach ($_POST['agenda_subject'] as $index => $subject) {
                    if (!empty($subject)) {
                        $subject_esc = mysqli_real_escape_string($conn, $subject);
                        $detail = isset($_POST['agenda_detail'][$index]) ? mysqli_real_escape_string($conn, $_POST['agenda_detail'][$index]) : '';
                        $prioritas = isset($_POST['agenda_prioritas'][$index]) ? mysqli_real_escape_string($conn, $_POST['agenda_prioritas'][$index]) : 'medium';
                        $urutan = $index + 1;
                        
                        $agenda_sql = "
                            INSERT INTO meeting_agenda (meeting_id, urutan, subject, detail_subject, prioritas) 
                            VALUES ($meeting_id, $urutan, '$subject_esc', '$detail', '$prioritas')
                        ";
                        
                        if (!mysqli_query($conn, $agenda_sql)) {
                            throw new Exception('Error inserting agenda: ' . mysqli_error($conn));
                        }
                    }
                }
            }
            
            // Commit transaction
            mysqli_commit($conn);
            $message = "Meeting berhasil dibuat dengan ID: " . $meeting_id;
            
            // Redirect to list page after success
            header("Location: meeting_list.php?success=1&id=" . $meeting_id);
            exit();
            
        } catch (Exception $e) {
            // Rollback transaction
            mysqli_rollback($conn);
            $error = "Error: " . $e->getMessage();
            error_log("Meeting Add Error: " . $e->getMessage());
        }
        
        // Re-enable autocommit
        mysqli_autocommit($conn, true);
    }
}

// Get data for dropdowns with error handling
$customers = [];
$suppliers = [];

try {
    // Get customers
    $customer_result = mysqli_query($conn, "SELECT id, nama FROM customer WHERE aktif = 1 ORDER BY nama");
    if ($customer_result) {
        while ($row = mysqli_fetch_assoc($customer_result)) {
            $customers[] = $row;
        }
    }
    
    // Get suppliers
    $supplier_result = mysqli_query($conn, "SELECT id, nama FROM supplier ORDER BY nama");
    if ($supplier_result) {
        while ($row = mysqli_fetch_assoc($supplier_result)) {
            $suppliers[] = $row;
        }
    }
} catch (Exception $e) {
    $error .= " Database query error: " . $e->getMessage();
}

// Handle mysqli errors
if (mysqli_error($conn)) {
    $error .= " MySQL Error: " . mysqli_error($conn);
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Tambah Meeting Baru</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/font-awesome.min.css">
    <style>
        .card-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        .agenda-item {
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 15px;
            background-color: #f8f9fa;
        }
    </style>
</head>
<body>

<?php 
// Include navbar with error handling
if (file_exists('navbar.php')) {
    include 'navbar.php'; 
}
?>

<div class="container mt-4">
    <div class="row">
        <div class="col-12">
            <div class="card shadow">
                <div class="card-header">
                    <h4 class="mb-0">
                        <i class="fa fa-plus-circle"></i> Tambah Meeting Baru
                    </h4>
                </div>
                <div class="card-body">
                    
                    <?php if ($message): ?>
                        <div class="alert alert-success alert-dismissible">
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            <?php echo htmlspecialchars($message); ?>
                        </div>
                    <?php endif; ?>
                    
                    <?php if ($error): ?>
                        <div class="alert alert-danger alert-dismissible">
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            <?php echo htmlspecialchars($error); ?>
                        </div>
                    <?php endif; ?>

                    <form method="POST" id="meetingForm">
                        
                        <!-- Meeting Basic Info -->
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label class="form-label"><strong>Tanggal Meeting *</strong></label>
                                <input type="date" name="tanggal" class="form-control" required value="<?php echo date('Y-m-d'); ?>">
                            </div>
                            <div class="col-md-3">
                                <label class="form-label"><strong>Waktu Mulai</strong></label>
                                <input type="time" name="waktu_mulai" class="form-control">
                            </div>
                            <div class="col-md-3">
                                <label class="form-label"><strong>Waktu Selesai</strong></label>
                                <input type="time" name="waktu_selesai" class="form-control">
                            </div>
                        </div>

                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label class="form-label"><strong>Tempat Meeting *</strong></label>
                                <input type="text" name="tempat" class="form-control" placeholder="Contoh: Ruang Rapat Utama, Online - Zoom" required>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label"><strong>Jenis Meeting</strong></label>
                                <select name="jenis_meeting" class="form-select">
                                    <option value="internal">Internal</option>
                                    <option value="external">External</option>
                                    <option value="customer">Customer</option>
                                    <option value="supplier">Supplier</option>
                                    <option value="planning">Planning</option>
                                    <option value="review">Review</option>
                                </select>
                            </div>
                        </div>

                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label class="form-label"><strong>Customer (Opsional)</strong></label>
                                <select name="customer_id" class="form-select">
                                    <option value="">Pilih Customer...</option>
                                    <?php foreach ($customers as $customer): ?>
                                        <option value="<?php echo htmlspecialchars($customer['id']); ?>">
                                            <?php echo htmlspecialchars($customer['nama']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label"><strong>Supplier (Opsional)</strong></label>
                                <select name="supplier_id" class="form-select">
                                    <option value="">Pilih Supplier...</option>
                                    <?php foreach ($suppliers as $supplier): ?>
                                        <option value="<?php echo htmlspecialchars($supplier['id']); ?>">
                                            <?php echo htmlspecialchars($supplier['nama']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>

                        <!-- Agenda Section -->
                        <hr>
                        <h5 class="mb-3"><i class="fa fa-list"></i> Agenda Meeting</h5>
                        <div id="agendaContainer">
                            <div class="agenda-item">
                                <div class="row mb-2">
                                    <div class="col-md-6">
                                        <label class="form-label"><strong>Subject Agenda *</strong></label>
                                        <input type="text" name="agenda_subject[]" class="form-control" placeholder="Contoh: Review Q3 Sales" required>
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label"><strong>Prioritas</strong></label>
                                        <select name="agenda_prioritas[]" class="form-select">
                                            <option value="low">Low</option>
                                            <option value="medium" selected>Medium</option>
                                            <option value="high">High</option>
                                            <option value="urgent">Urgent</option>
                                        </select>
                                    </div>
                                </div>
                                <div class="row">
                                    <div class="col-12">
                                        <label class="form-label"><strong>Detail Agenda / Hasil Meeting</strong></label>
                                        <textarea name="agenda_detail[]" class="form-control" rows="4" placeholder="Detail pembahasan agenda dan hasil meeting..."></textarea>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <button type="button" class="btn btn-outline-primary btn-sm" onclick="addAgenda()">
                                <i class="fa fa-plus"></i> Tambah Agenda
                            </button>
                        </div>

                        <!-- Submit Button -->
                        <hr>
                        <div class="d-flex justify-content-between">
                            <a href="meeting_list.php" class="btn btn-secondary">
                                <i class="fa fa-arrow-left"></i> Kembali ke Daftar
                            </a>
                            <button type="submit" class="btn btn-success" <?php echo !$current_user_id ? 'disabled' : ''; ?>>
                                <i class="fa fa-save"></i> Simpan Meeting
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
function addAgenda() {
    const container = document.getElementById('agendaContainer');
    const agendaHtml = `
        <div class="agenda-item">
            <div class="row mb-2">
                <div class="col-md-5">
                    <label class="form-label"><strong>Subject Agenda *</strong></label>
                    <input type="text" name="agenda_subject[]" class="form-control" placeholder="Contoh: Review Q3 Sales" required>
                </div>
                <div class="col-md-5">
                    <label class="form-label"><strong>Prioritas</strong></label>
                    <select name="agenda_prioritas[]" class="form-select">
                        <option value="low">Low</option>
                        <option value="medium" selected>Medium</option>
                        <option value="high">High</option>
                        <option value="urgent">Urgent</option>
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">&nbsp;</label>
                    <button type="button" class="btn btn-danger btn-sm d-block" onclick="removeAgenda(this)">
                        <i class="fa fa-trash"></i> Hapus
                    </button>
                </div>
            </div>
            <div class="row">
                <div class="col-12">
                    <label class="form-label"><strong>Detail Agenda / Hasil Meeting</strong></label>
                    <textarea name="agenda_detail[]" class="form-control" rows="4" placeholder="Detail pembahasan agenda dan hasil meeting..."></textarea>
                </div>
            </div>
        </div>
    `;
    container.insertAdjacentHTML('beforeend', agendaHtml);
}

function removeAgenda(button) {
    button.closest('.agenda-item').remove();
}

// Set minimum date to today
document.addEventListener('DOMContentLoaded', function() {
    const dateInput = document.querySelector('input[name="tanggal"]');
    const today = new Date().toISOString().split('T')[0];
    dateInput.min = today;
});

// Form validation
document.getElementById('meetingForm').addEventListener('submit', function(e) {
    const agendaSubjects = document.querySelectorAll('input[name="agenda_subject[]"]');
    let hasValidAgenda = false;
    
    agendaSubjects.forEach(function(input) {
        if (input.value.trim() !== '') {
            hasValidAgenda = true;
        }
    });
    
    if (!hasValidAgenda) {
        e.preventDefault();
        alert('Minimal harus ada 1 agenda meeting!');
        return false;
    }
});
</script>

</body>
</html>