<?php
include 'config.php';
include 'session.php';
include 'navbar.php';

// Filter parameters
$filter_supplier = isset($_GET['supplier']) ? (int)$_GET['supplier'] : 0;
$filter_tanggal_dari = isset($_GET['dari']) ? $_GET['dari'] : '';
$filter_tanggal_sampai = isset($_GET['sampai']) ? $_GET['sampai'] : '';
$filter_status = isset($_GET['status']) ? $_GET['status'] : '';

// Build WHERE clause
$where_conditions = array();
$params = array();
$param_types = '';

if ($filter_supplier > 0) {
    $where_conditions[] = "p.supplier_id = ?";
    $params[] = $filter_supplier;
    $param_types .= 'i';
}

if (!empty($filter_tanggal_dari)) {
    $where_conditions[] = "p.tanggal >= ?";
    $params[] = $filter_tanggal_dari;
    $param_types .= 's';
}

if (!empty($filter_tanggal_sampai)) {
    $where_conditions[] = "p.tanggal <= ?";
    $params[] = $filter_tanggal_sampai;
    $param_types .= 's';
}

if (!empty($filter_status)) {
    $where_conditions[] = "p.status = ?";
    $params[] = $filter_status;
    $param_types .= 's';
}

$where_clause = '';
if (count($where_conditions) > 0) {
    $where_clause = 'WHERE ' . implode(' AND ', $where_conditions);
}

// Main query
$query = "
    SELECT 
        p.id,
        p.no_pembelian,
        p.tanggal,
        s.nama as supplier_nama,
        p.total_amount,
        p.status,
        p.keterangan,
        COUNT(pd.id) as jumlah_item,
        SUM(pd.qty_beli) as total_qty_beli,
        SUM(pd.qty_tersisa) as total_qty_tersisa,
        CASE 
            WHEN SUM(pd.qty_tersisa) = 0 THEN 'Habis'
            WHEN SUM(pd.qty_tersisa) = SUM(pd.qty_beli) THEN 'Belum Terpakai'
            ELSE 'Sebagian Terpakai'
        END as status_stock,
        p.created_at
    FROM pembelian p
    LEFT JOIN supplier s ON p.supplier_id = s.id
    LEFT JOIN pembelian_detail pd ON p.id = pd.pembelian_id
    $where_clause
    GROUP BY p.id
    ORDER BY p.tanggal DESC, p.id DESC
";

if (!empty($params)) {
    $stmt = mysqli_prepare($conn, $query);
    mysqli_stmt_bind_param($stmt, $param_types, ...$params);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
} else {
    $result = mysqli_query($conn, $query);
}

// Ambil daftar supplier untuk filter
$supplier_list = array();
$q_supplier = mysqli_query($conn, "SELECT id, nama FROM supplier ORDER BY nama");
if ($q_supplier) {
    while ($row = mysqli_fetch_assoc($q_supplier)) {
        $supplier_list[] = $row;
    }
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Daftar Pembelian - SBA System</title>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background: #f8f9fa;
            color: #333;
        }
        
        .container { 
            max-width: 1400px; 
            margin: 0 auto; 
            padding: 20px; 
        }
        
        .btn { 
            border-radius: 6px; 
            padding: 8px 16px; 
            font-weight: 500; 
            text-decoration: none; 
        }
        
        .table { 
            background: white; 
            border-radius: 8px; 
            overflow: hidden;
        }
        
        .badge { 
            padding: 4px 8px; 
            border-radius: 4px; 
            font-size: 11px; 
        }
        
        .filter-card {
            background: white;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .summary-cards {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }
        
        .summary-card {
            background: white;
            border-radius: 8px;
            padding: 20px;
            text-align: center;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .summary-card h3 {
            font-size: 24px;
            margin: 0;
            color: #007bff;
        }
        
        .summary-card p {
            margin: 5px 0 0 0;
            color: #666;
            font-size: 14px;
        }
        
        .stock-progress {
            width: 100%;
            height: 6px;
            background: #e9ecef;
            border-radius: 3px;
            overflow: hidden;
        }
        
        .stock-progress-bar {
            height: 100%;
            background: #28a745;
            transition: width 0.3s ease;
        }
        
        .stock-progress-bar.warning {
            background: #ffc107;
        }
        
        .stock-progress-bar.danger {
            background: #dc3545;
        }
    </style>
</head>
<body>

<div class="container mt-4">
    <div class="d-flex justify-content-between align-items-center mb-3">
        <h3>Daftar Pembelian</h3>
        <div>
            <a href="pembelian_form.php" class="btn btn-primary">+ Tambah Pembelian</a>
            <a href="penjualan_list.php" class="btn btn-info">Daftar Penjualan</a>
        </div>
    </div>

    <!-- Summary Cards -->
    <?php
    $summary_query = "
        SELECT 
            COUNT(*) as total_pembelian,
            SUM(total_amount) as total_nilai,
            SUM(CASE WHEN status = 'confirmed' THEN 1 ELSE 0 END) as confirmed_count,
            SUM(CASE WHEN status = 'draft' THEN 1 ELSE 0 END) as draft_count
        FROM pembelian p
        $where_clause
    ";
    
    if (!empty($params)) {
        $stmt_summary = mysqli_prepare($conn, $summary_query);
        mysqli_stmt_bind_param($stmt_summary, $param_types, ...$params);
        mysqli_stmt_execute($stmt_summary);
        $summary_result = mysqli_stmt_get_result($stmt_summary);
    } else {
        $summary_result = mysqli_query($conn, $summary_query);
    }
    
    $summary = mysqli_fetch_assoc($summary_result);
    ?>
    
    <div class="summary-cards">
        <div class="summary-card">
            <h3><?php echo number_format($summary['total_pembelian']); ?></h3>
            <p>Total Pembelian</p>
        </div>
        <div class="summary-card">
            <h3>Rp <?php echo number_format($summary['total_nilai'], 0, ',', '.'); ?></h3>
            <p>Total Nilai</p>
        </div>
        <div class="summary-card">
            <h3><?php echo number_format($summary['confirmed_count']); ?></h3>
            <p>Confirmed</p>
        </div>
        <div class="summary-card">
            <h3><?php echo number_format($summary['draft_count']); ?></h3>
            <p>Draft</p>
        </div>
    </div>

    <!-- Filter Card -->
    <div class="filter-card">
        <form method="GET" action="">
            <div class="row">
                <div class="col-md-3">
                    <label class="form-label">Supplier</label>
                    <select name="supplier" class="form-select">
                        <option value="">-- Semua Supplier --</option>
                        <?php foreach ($supplier_list as $supplier): ?>
                        <option value="<?php echo $supplier['id']; ?>" 
                                <?php echo $filter_supplier == $supplier['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($supplier['nama']); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Dari Tanggal</label>
                    <input type="date" name="dari" class="form-control" value="<?php echo $filter_tanggal_dari; ?>">
                </div>
                <div class="col-md-2">
                    <label class="form-label">Sampai Tanggal</label>
                    <input type="date" name="sampai" class="form-control" value="<?php echo $filter_tanggal_sampai; ?>">
                </div>
                <div class="col-md-2">
                    <label class="form-label">Status</label>
                    <select name="status" class="form-select">
                        <option value="">-- Semua Status --</option>
                        <option value="confirmed" <?php echo $filter_status == 'confirmed' ? 'selected' : ''; ?>>Confirmed</option>
                        <option value="draft" <?php echo $filter_status == 'draft' ? 'selected' : ''; ?>>Draft</option>
                    </select>
                </div>
                <div class="col-md-3 d-flex align-items-end">
                    <button type="submit" class="btn btn-primary me-2">Filter</button>
                    <a href="?" class="btn btn-secondary">Reset</a>
                </div>
            </div>
        </form>
    </div>

    <!-- Data Table -->
    <div class="table-responsive">
        <table class="table table-hover">
            <thead class="table-dark">
                <tr>
                    <th>No. Pembelian</th>
                    <th>Tanggal</th>
                    <th>Supplier</th>
                    <th>Items</th>
                    <th>Total Qty</th>
                    <th>Stock Status</th>
                    <th>Total Nilai</th>
                    <th>Status</th>
                    <th>Aksi</th>
                </tr>
            </thead>
            <tbody>
                <?php if ($result && mysqli_num_rows($result) > 0): ?>
                    <?php while ($row = mysqli_fetch_assoc($result)): ?>
                    <tr>
                        <td>
                            <strong><?php echo htmlspecialchars($row['no_pembelian']); ?></strong>
                            <?php if (!empty($row['keterangan'])): ?>
                                <br><small class="text-muted"><?php echo htmlspecialchars($row['keterangan']); ?></small>
                            <?php endif; ?>
                        </td>
                        <td><?php echo date('d/m/Y', strtotime($row['tanggal'])); ?></td>
                        <td><?php echo htmlspecialchars($row['supplier_nama']); ?></td>
                        <td>
                            <span class="badge bg-info"><?php echo $row['jumlah_item']; ?> items</span>
                        </td>
                        <td>
                            <div><?php echo number_format($row['total_qty_beli'], 2); ?></div>
                            <small class="text-muted">Tersisa: <?php echo number_format($row['total_qty_tersisa'], 2); ?></small>
                        </td>
                        <td>
                            <?php
                            $usage_percent = 0;
                            if ($row['total_qty_beli'] > 0) {
                                $usage_percent = (($row['total_qty_beli'] - $row['total_qty_tersisa']) / $row['total_qty_beli']) * 100;
                            }
                            
                            $status_class = 'success';
                            if ($usage_percent >= 80) {
                                $status_class = 'warning';
                            }
                            if ($usage_percent >= 95) {
                                $status_class = 'danger';
                            }
                            ?>
                            <div class="mb-1">
                                <span class="badge bg-<?php echo $status_class; ?>"><?php echo $row['status_stock']; ?></span>
                            </div>
                            <div class="stock-progress">
                                <div class="stock-progress-bar <?php echo $status_class; ?>" 
                                     style="width: <?php echo $usage_percent; ?>%"></div>
                            </div>
                            <small class="text-muted"><?php echo number_format($usage_percent, 1); ?>% terpakai</small>
                        </td>
                        <td>
                            <strong>Rp <?php echo number_format($row['total_amount'], 0, ',', '.'); ?></strong>
                        </td>
                        <td>
                            <?php if ($row['status'] == 'confirmed'): ?>
                                <span class="badge bg-success">Confirmed</span>
                            <?php elseif ($row['status'] == 'draft'): ?>
                                <span class="badge bg-warning">Draft</span>
                            <?php else: ?>
                                <span class="badge bg-secondary"><?php echo ucfirst($row['status']); ?></span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <div class="btn-group btn-group-sm">
                                <a href="pembelian_form.php?id=<?php echo $row['id']; ?>" 
                                   class="btn btn-outline-primary" title="Edit">
                                    ✏️
                                </a>
                                <a href="pembelian_detail.php?id=<?php echo $row['id']; ?>" 
                                   class="btn btn-outline-info" title="Detail">
                                    👁️
                                </a>
                                <a href="pembelian_cetak.php?id=<?php echo $row['id']; ?>" 
                                   target="_blank" class="btn btn-outline-danger" title="Cetak PDF">
                                    🖨️
                                </a>
                                <?php if ($row['status'] == 'draft'): ?>
                                <button onclick="hapusPembelian(<?php echo $row['id']; ?>)" 
                                        class="btn btn-outline-danger" title="Hapus">
                                    🗑️
                                </button>
                                <?php endif; ?>
                            </div>
                        </td>
                    </tr>
                    <?php endwhile; ?>
                <?php else: ?>
                    <tr>
                        <td colspan="9" class="text-center py-4">
                            <div class="text-muted">
                                <i class="fas fa-inbox fa-3x mb-3"></i>
                                <h5>Tidak ada data pembelian</h5>
                                <p>Silakan tambah pembelian baru atau ubah filter pencarian.</p>
                            </div>
                        </td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <!-- Pagination -->
    <?php
    // Simple pagination (optional - add LIMIT/OFFSET to query if needed)
    $total_query = str_replace("SELECT p.id, p.no_pembelian, p.tanggal, s.nama as supplier_nama, p.total_amount, p.status, p.keterangan, COUNT(pd.id) as jumlah_item, SUM(pd.qty_beli) as total_qty_beli, SUM(pd.qty_tersisa) as total_qty_tersisa, CASE WHEN SUM(pd.qty_tersisa) = 0 THEN 'Habis' WHEN SUM(pd.qty_tersisa) = SUM(pd.qty_beli) THEN 'Belum Terpakai' ELSE 'Sebagian Terpakai' END as status_stock, p.created_at", "SELECT COUNT(DISTINCT p.id)", $query);
    $total_query = str_replace("ORDER BY p.tanggal DESC, p.id DESC", "", $total_query);
    
    if (!empty($params)) {
        $stmt_total = mysqli_prepare($conn, $total_query);
        mysqli_stmt_bind_param($stmt_total, $param_types, ...$params);
        mysqli_stmt_execute($stmt_total);
        $total_result = mysqli_stmt_get_result($stmt_total);
    } else {
        $total_result = mysqli_query($conn, $total_query);
    }
    
    $total_records = mysqli_fetch_row($total_result)[0];
    ?>
    
    <div class="d-flex justify-content-between align-items-center mt-3">
        <div class="text-muted">
            Menampilkan <?php echo number_format($total_records); ?> data pembelian
        </div>
        <div>
            <a href="pembelian_export.php?<?php echo http_build_query($_GET); ?>" 
               class="btn btn-success btn-sm">
                📊 Export Excel
            </a>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

<script>
function hapusPembelian(id) {
    if (confirm('Yakin ingin menghapus pembelian ini?\n\nPerhatian: Data yang sudah terhubung dengan penjualan tidak dapat dihapus!')) {
        fetch('pembelian_hapus.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: 'id=' + id
        })
        .then(response => response.text())
        .then(data => {
            if (data.trim() === 'OK') {
                alert('Pembelian berhasil dihapus!');
                location.reload();
            } else {
                alert('Gagal menghapus: ' + data);
            }
        })
        .catch(error => {
            alert('Error: ' + error);
        });
    }
}

// Auto-refresh setiap 5 menit untuk update stock real-time
setTimeout(function() {
    location.reload();
}, 300000);
</script>

</body>
</html>