<?php
include 'config.php';
include 'session.php';

// Handle POST requests (Add/Update)
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $id = isset($_POST['relasi_id']) && $_POST['relasi_id'] !== '' ? (int)$_POST['relasi_id'] : 0;
    $produk_id = isset($_POST['produk_id']) ? (int)$_POST['produk_id'] : 0;
    $supplier_id = isset($_POST['supplier_id']) ? (int)$_POST['supplier_id'] : 0;
    $harga = isset($_POST['harga']) ? (float)$_POST['harga'] : 0;

    // Validasi input
    if ($produk_id <= 0) {
        $_SESSION['error_message'] = "Produk harus dipilih!";
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }
    
    if ($supplier_id <= 0) {
        $_SESSION['error_message'] = "Supplier harus dipilih!";
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }
    
    if ($harga <= 0) {
        $_SESSION['error_message'] = "Harga harus lebih dari 0!";
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }

    try {
        // Cek duplikat kombinasi produk_id + supplier_id
        $cek = $conn->prepare("SELECT id FROM produk_supplier WHERE produk_id=? AND supplier_id=?");
        $cek->bind_param("ii", $produk_id, $supplier_id);
        $cek->execute();
        $cek->store_result();
        $cek_id = null;
        if ($cek->num_rows > 0) {
            $cek->bind_result($cek_id);
            $cek->fetch();
        }

        if ($id > 0) {
            // UPDATE
            if ($cek_id && $cek_id != $id) {
                $_SESSION['error_message'] = "Kombinasi produk dan supplier sudah ada!";
                $cek->close();
                header("Location: " . $_SERVER['PHP_SELF']);
                exit;
            } else {
                $stmt = $conn->prepare("UPDATE produk_supplier SET produk_id=?, supplier_id=?, harga=?, updated_at=NOW() WHERE id=?");
                $stmt->bind_param("iidi", $produk_id, $supplier_id, $harga, $id);
                $message = "Data berhasil diupdate!";
            }
        } else {
            // INSERT
            if ($cek->num_rows > 0) {
                // Update harga jika kombinasi sudah ada
                $stmt = $conn->prepare("UPDATE produk_supplier SET harga=?, updated_at=NOW() WHERE produk_id=? AND supplier_id=?");
                $stmt->bind_param("dii", $harga, $produk_id, $supplier_id);
                $message = "Harga berhasil diupdate!";
            } else {
                $stmt = $conn->prepare("INSERT INTO produk_supplier (produk_id, supplier_id, harga, updated_at) VALUES (?, ?, ?, NOW())");
                $stmt->bind_param("iid", $produk_id, $supplier_id, $harga);
                $message = "Data berhasil ditambahkan!";
            }
        }

        if ($stmt->execute()) {
            $_SESSION['success_message'] = $message;
        } else {
            $_SESSION['error_message'] = "Gagal menyimpan data: " . $stmt->error;
        }
        $stmt->close();
        $cek->close();

    } catch (Exception $e) {
        $_SESSION['error_message'] = "Error: " . $e->getMessage();
        error_log("Master Produk Supplier Error: " . $e->getMessage());
    }

    header("Location: " . $_SERVER['PHP_SELF']);
    exit;
}

// Handle DELETE requests
if (isset($_GET['hapus']) && is_numeric($_GET['hapus'])) {
    $id = (int)$_GET['hapus'];
    
    try {
        $stmt = $conn->prepare("DELETE FROM produk_supplier WHERE id = ?");
        $stmt->bind_param("i", $id);
        
        if ($stmt->execute()) {
            $_SESSION['success_message'] = "Data berhasil dihapus!";
        } else {
            $_SESSION['error_message'] = "Gagal menghapus data!";
        }
        $stmt->close();
        
    } catch (Exception $e) {
        $_SESSION['error_message'] = "Error saat menghapus: " . $e->getMessage();
        error_log("Master Produk Supplier Delete Error: " . $e->getMessage());
    }
    
    header("Location: " . $_SERVER['PHP_SELF']);
    exit;
}

// Get filter parameters
$filter_produk_nama = isset($_GET['produk_nama']) ? trim($_GET['produk_nama']) : '';
$filter_supplier_nama = isset($_GET['supplier_nama']) ? trim($_GET['supplier_nama']) : '';
$show_last_update = isset($_GET['show_last_update']);

// Build WHERE clause
$where = [];
if ($filter_produk_nama !== '') {
    $where[] = "pi.nama LIKE '%" . mysqli_real_escape_string($conn, $filter_produk_nama) . "%'";
}
if ($filter_supplier_nama !== '') {
    $where[] = "s.nama LIKE '%" . mysqli_real_escape_string($conn, $filter_supplier_nama) . "%'";
}

$whereSql = $where ? ("WHERE " . implode(" AND ", $where)) : "";

// Get data with error handling
try {
    $query = "SELECT ps.*, pi.nama AS produk_nama, s.nama AS supplier_nama, s.kontak, ps.produk_id, ps.supplier_id
              FROM produk_supplier ps
              JOIN produk_interest pi ON pi.id = ps.produk_id
              JOIN supplier s ON s.id = ps.supplier_id
              $whereSql
              ORDER BY ps.harga ASC";
    
    $data = mysqli_query($conn, $query);
    if (!$data) {
        throw new Exception("Error fetching data: " . mysqli_error($conn));
    }
    $totalData = mysqli_num_rows($data);
} catch (Exception $e) {
    $_SESSION['error_message'] = "Error mengambil data: " . $e->getMessage();
    $totalData = 0;
    $data = false;
}

// Get dropdown data
try {
    $produk = mysqli_query($conn, "SELECT * FROM produk_interest ORDER BY nama ASC");
    $supplier = mysqli_query($conn, "SELECT * FROM supplier ORDER BY nama ASC");
    if (!$produk || !$supplier) {
        throw new Exception("Error fetching dropdown data");
    }
} catch (Exception $e) {
    $produk = false;
    $supplier = false;
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Master Produk Supplier</title>
    <link rel="icon" type="image/x-icon" href="/images/favicon.ico">
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <style>
        .btn-action {
            padding: 4px 8px;
            border: none;
            margin: 0 2px;
            cursor: pointer;
            border-radius: 3px;
            display: inline-block;
            font-size: 11px;
            color: white;
            text-decoration: none;
        }
        .btn-edit {
            background-color: #ffc107;
            color: #000;
        }
        .btn-edit:hover {
            background-color: #e0a800;
            color: #000;
        }
        .btn-delete {
            background-color: #dc3545;
            color: #fff;
        }
        .btn-delete:hover {
            background-color: #c82333;
            color: #fff;
        }
        .whatsapp-link {
            color: #25d366;
            text-decoration: none;
        }
        .whatsapp-link:hover {
            color: #128c7e;
            text-decoration: underline;
        }
        .select2-container {
            width: 100% !important;
        }
        .table td {
            vertical-align: middle;
        }
    </style>
</head>
<body>

<!-- Include Menu Navigation -->
<?php 
// Set variabel untuk navbar
if (!isset($jabatan_relasi)) {
    $jabatan_relasi = $_SESSION['jabatan'] ?? '';
}
include 'navbar.php'; 
?>

<div class="container-fluid py-4">
    
    <!-- Page Header -->
    <div class="row mb-4">
        <div class="col">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h3 class="mb-1 text-dark">
                        <i class="fa fa-link text-primary"></i> Master Produk Supplier
                    </h3>
                    <p class="text-muted mb-0">Kelola relasi produk dengan supplier dan harga</p>
                </div>
                <div>
                    <span class="badge bg-primary fs-6">
                        <i class="fa fa-database"></i> <?= $totalData ?> Relasi
                    </span>
                </div>
            </div>
        </div>
    </div>

    <!-- Alert Messages -->
    <?php if (isset($_SESSION['success_message'])): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fa fa-check-circle"></i> <?= htmlspecialchars($_SESSION['success_message']) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php unset($_SESSION['success_message']); ?>
    <?php endif; ?>

    <?php if (isset($_SESSION['error_message'])): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fa fa-exclamation-circle"></i> <?= htmlspecialchars($_SESSION['error_message']) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php unset($_SESSION['error_message']); ?>
    <?php endif; ?>

    <!-- Filter Section -->
    <div class="card shadow-sm mb-4">
        <div class="card-header bg-light">
            <h6 class="mb-0">
                <i class="fa fa-filter text-secondary"></i> Filter & Pencarian Data
            </h6>
        </div>
        <div class="card-body">
            <form method="get" class="row g-3">
                <div class="col-md-4">
                    <label class="form-label small">Nama Produk</label>
                    <input type="text" name="produk_nama" class="form-control" 
                           placeholder="Cari produk..." 
                           value="<?= htmlspecialchars($filter_produk_nama) ?>">
                </div>
                <div class="col-md-4">
                    <label class="form-label small">Nama Supplier</label>
                    <input type="text" name="supplier_nama" class="form-control" 
                           placeholder="Cari supplier..." 
                           value="<?= htmlspecialchars($filter_supplier_nama) ?>">
                </div>
                <div class="col-md-4">
                    <label class="form-label small">Opsi Tampilan</label>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="show_last_update" 
                               id="showLastUpdate" value="1" <?= $show_last_update ? 'checked' : '' ?>>
                        <label class="form-check-label" for="showLastUpdate">
                            <i class="fa fa-clock"></i> Tampilkan Last Update
                        </label>
                    </div>
                </div>
                <div class="col-12">
                    <button type="submit" class="btn btn-secondary me-2">
                        <i class="fa fa-filter"></i> Terapkan Filter
                    </button>
                    <a href="<?= $_SERVER['PHP_SELF'] ?>" class="btn btn-outline-secondary">
                        <i class="fa fa-refresh"></i> Reset Filter
                    </a>
                </div>
            </form>
        </div>
    </div>

    <!-- Action Buttons -->
    <div class="row mb-3">
        <div class="col-md-6">
            <button class="btn btn-success" onclick="openModal()">
                <i class="fa fa-plus"></i> Tambah Relasi Baru
            </button>
        </div>
        <div class="col-md-6 text-end">
            <?php if ($filter_produk_nama || $filter_supplier_nama): ?>
                <span class="badge bg-info">
                    <i class="fa fa-filter"></i> Filter Aktif
                </span>
            <?php endif; ?>
        </div>
    </div>

    <!-- Data Table -->
    <div class="row">
        <div class="col">
            <div class="card shadow-sm">
                <div class="card-header bg-dark text-white">
                    <h5 class="mb-0">
                        <i class="fa fa-list"></i> Daftar Relasi Produk Supplier
                    </h5>
                </div>
                <div class="card-body p-0">
                    <?php if ($data && $totalData > 0): ?>
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead class="table-light">
                                    <tr>
                                        <th width="50" class="text-center">#</th>
                                        <th>Produk</th>
                                        <th>Supplier</th>
                                        <th width="150">Kontak</th>
                                        <th width="120" class="text-end">Harga</th>
                                        <?php if ($show_last_update): ?>
                                            <th width="150" class="text-center">Last Update</th>
                                        <?php endif; ?>
                                        <th width="120" class="text-center">Aksi</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php 
                                    $no = 1;
                                    mysqli_data_seek($data, 0); // Reset pointer
                                    while ($row = mysqli_fetch_assoc($data)): 
                                    ?>
                                        <tr>
                                            <td class="text-center">
                                                <span class="badge bg-secondary"><?= $no++ ?></span>
                                            </td>
                                            <td>
                                                <strong><?= htmlspecialchars($row['produk_nama']) ?></strong>
                                            </td>
                                            <td>
                                                <span class="text-muted"><?= htmlspecialchars($row['supplier_nama']) ?></span>
                                            </td>
                                            <td>
                                                <?php if (!empty($row['kontak'])): ?>
                                                    <a href="https://wa.me/<?= preg_replace('/[^0-9]/', '', $row['kontak']) ?>" 
                                                       target="_blank" class="whatsapp-link" title="Chat via WhatsApp">
                                                        <i class="fa fa-whatsapp"></i> <?= htmlspecialchars($row['kontak']) ?>
                                                    </a>
                                                <?php else: ?>
                                                    <em class="text-muted">-</em>
                                                <?php endif; ?>
                                            </td>
                                            <td class="text-end">
                                                <strong class="text-success">
                                                    Rp <?= number_format($row['harga'], 0, ',', '.') ?>
                                                </strong>
                                            </td>
                                            <?php if ($show_last_update): ?>
                                                <td class="text-center">
                                                    <small class="text-muted">
                                                        <?= !empty($row['updated_at']) ? date('d/m/Y H:i', strtotime($row['updated_at'])) : '-' ?>
                                                    </small>
                                                </td>
                                            <?php endif; ?>
                                            <td class="text-center">
                                                <button class="btn-action btn-edit" 
                                                        onclick='editModal(<?= json_encode($row, JSON_HEX_APOS | JSON_HEX_QUOT) ?>)'
                                                        title="Edit Relasi">
                                                    Edit
                                                </button>
                                                <button class="btn-action btn-delete" 
                                                        onclick="confirmDelete(<?= (int)$row['id'] ?>, '<?= htmlspecialchars($row['produk_nama'], ENT_QUOTES) ?>', '<?= htmlspecialchars($row['supplier_nama'], ENT_QUOTES) ?>')"
                                                        title="Hapus Relasi">
                                                    Hapus
                                                </button>
                                            </td>
                                        </tr>
                                    <?php endwhile; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php else: ?>
                        <div class="text-center py-5">
                            <i class="fa fa-search fa-3x text-muted mb-3"></i>
                            <h5 class="text-muted">
                                <?php if ($data === false): ?>
                                    Error mengambil data
                                <?php elseif ($filter_produk_nama || $filter_supplier_nama): ?>
                                    Tidak ada data sesuai filter
                                <?php else: ?>
                                    Belum ada relasi produk supplier
                                <?php endif; ?>
                            </h5>
                            <p class="text-muted">
                                <?php if ($data === false): ?>
                                    Silakan refresh halaman atau hubungi administrator
                                <?php elseif ($filter_produk_nama || $filter_supplier_nama): ?>
                                    Coba ubah kriteria pencarian atau reset filter
                                <?php else: ?>
                                    Klik tombol "Tambah Relasi Baru" untuk menambah data pertama
                                <?php endif; ?>
                            </p>
                            <?php if ($data !== false && !($filter_produk_nama || $filter_supplier_nama)): ?>
                                <button class="btn btn-success" onclick="openModal()">
                                    <i class="fa fa-plus"></i> Tambah Sekarang
                                </button>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                </div>
                <?php if ($data && $totalData > 0): ?>
                    <div class="card-footer bg-light">
                        <small class="text-muted">
                            <i class="fa fa-info-circle"></i> 
                            Menampilkan <strong><?= $totalData ?></strong> relasi produk supplier
                            <?php if ($filter_produk_nama || $filter_supplier_nama): ?>
                                dari hasil filter
                            <?php endif; ?>
                        </small>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

</div>

<!-- Modal Add/Edit -->
<div class="modal fade" id="relasiModal" tabindex="-1" aria-labelledby="modalTitle" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <form method="post" class="modal-content" onsubmit="return validateForm()">
            <div class="modal-header">
                <h5 class="modal-title" id="modalTitle">
                    <i class="fa fa-plus"></i> Tambah Relasi
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <input type="hidden" name="relasi_id" id="relasiId">
                
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="produkSelect" class="form-label">
                                <i class="fa fa-box"></i> Produk <span class="text-danger">*</span>
                            </label>
                            <select name="produk_id" id="produkSelect" class="form-select select2" required>
                                <option value="">-- Pilih Produk --</option>
                                <?php if ($produk): ?>
                                    <?php mysqli_data_seek($produk, 0); while ($p = mysqli_fetch_assoc($produk)): ?>
                                        <option value="<?= (int)$p['id'] ?>"><?= htmlspecialchars($p['nama']) ?></option>
                                    <?php endwhile; ?>
                                <?php endif; ?>
                            </select>
                            <div class="invalid-feedback" id="produkError"></div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="supplierSelect" class="form-label">
                                <i class="fa fa-truck"></i> Supplier <span class="text-danger">*</span>
                            </label>
                            <select name="supplier_id" id="supplierSelect" class="form-select select2" required>
                                <option value="">-- Pilih Supplier --</option>
                                <?php if ($supplier): ?>
                                    <?php mysqli_data_seek($supplier, 0); while ($s = mysqli_fetch_assoc($supplier)): ?>
                                        <option value="<?= (int)$s['id'] ?>"><?= htmlspecialchars($s['nama']) ?></option>
                                    <?php endwhile; ?>
                                <?php endif; ?>
                            </select>
                            <div class="invalid-feedback" id="supplierError"></div>
                        </div>
                    </div>
                </div>
                
                <div class="mb-3">
                    <label for="hargaInput" class="form-label">
                        <i class="fa fa-money-bill"></i> Harga <span class="text-danger">*</span>
                    </label>
                    <div class="input-group">
                        <span class="input-group-text">Rp</span>
                        <input type="number" class="form-control" name="harga" id="hargaInput" 
                               placeholder="0" required min="1" step="0.01">
                    </div>
                    <div class="form-text">Masukkan harga dalam Rupiah</div>
                    <div class="invalid-feedback" id="hargaError"></div>
                </div>
                
                <div class="alert alert-info" role="alert">
                    <i class="fa fa-info-circle"></i>
                    <strong>Catatan:</strong> Jika kombinasi produk dan supplier sudah ada, harga akan diupdate.
                </div>
            </div>
            <div class="modal-footer">
                <button type="submit" class="btn btn-success" id="submitBtn">
                    <i class="fa fa-save"></i> Simpan Relasi
                </button>
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <i class="fa fa-times"></i> Batal
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Confirmation Modal -->
<div class="modal fade" id="confirmModal" tabindex="-1">
    <div class="modal-dialog modal-sm">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="fa fa-exclamation-triangle text-warning"></i> Konfirmasi
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>Yakin ingin menghapus relasi ini?</p>
                <p class="text-muted"><strong id="confirmItemName"></strong></p>
            </div>
            <div class="modal-footer">
                <a href="#" id="confirmDeleteBtn" class="btn btn-danger">
                    <i class="fa fa-trash"></i> Ya, Hapus
                </a>
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
            </div>
        </div>
    </div>
</div>

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>

<script>
// Validation function
function validateForm() {
    const produk = document.getElementById('produkSelect');
    const supplier = document.getElementById('supplierSelect');
    const harga = document.getElementById('hargaInput');
    const submitBtn = document.getElementById('submitBtn');
    
    // Reset validation
    [produk, supplier, harga].forEach(field => {
        field.classList.remove('is-invalid');
        const errorElement = document.getElementById(field.id.replace('Select', '').replace('Input', '') + 'Error');
        if (errorElement) errorElement.textContent = '';
    });
    
    let isValid = true;
    
    // Validate produk
    if (!produk.value) {
        produk.classList.add('is-invalid');
        document.getElementById('produkError').textContent = 'Produk harus dipilih!';
        if (isValid) produk.focus();
        isValid = false;
    }
    
    // Validate supplier
    if (!supplier.value) {
        supplier.classList.add('is-invalid');
        document.getElementById('supplierError').textContent = 'Supplier harus dipilih!';
        if (isValid) supplier.focus();
        isValid = false;
    }
    
    // Validate harga
    if (!harga.value || parseFloat(harga.value) <= 0) {
        harga.classList.add('is-invalid');
        document.getElementById('hargaError').textContent = 'Harga harus lebih dari 0!';
        if (isValid) harga.focus();
        isValid = false;
    }
    
    if (isValid) {
        // Disable submit button to prevent double submission
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<i class="fa fa-spinner fa-spin"></i> Menyimpan...';
    }
    
    return isValid;
}

// Open add modal
function openModal() {
    document.getElementById('modalTitle').innerHTML = '<i class="fa fa-plus"></i> Tambah Relasi';
    document.getElementById('relasiId').value = '';
    $('#produkSelect').val('').trigger('change');
    $('#supplierSelect').val('').trigger('change');
    document.getElementById('hargaInput').value = '';
    
    // Reset validation
    ['produkSelect', 'supplierSelect', 'hargaInput'].forEach(fieldId => {
        const field = document.getElementById(fieldId);
        if (field) field.classList.remove('is-invalid');
    });
    
    const submitBtn = document.getElementById('submitBtn');
    if (submitBtn) {
        submitBtn.disabled = false;
        submitBtn.innerHTML = '<i class="fa fa-save"></i> Simpan Relasi';
    }
    
    const modal = new bootstrap.Modal(document.getElementById('relasiModal'));
    modal.show();
}

// Edit modal
function editModal(data) {
    document.getElementById('modalTitle').innerHTML = '<i class="fa fa-edit"></i> Edit Relasi';
    document.getElementById('relasiId').value = data.id;
    $('#produkSelect').val(data.produk_id).trigger('change');
    $('#supplierSelect').val(data.supplier_id).trigger('change');
    document.getElementById('hargaInput').value = data.harga;
    
    // Reset validation
    ['produkSelect', 'supplierSelect', 'hargaInput'].forEach(fieldId => {
        const field = document.getElementById(fieldId);
        if (field) field.classList.remove('is-invalid');
    });
    
    const submitBtn = document.getElementById('submitBtn');
    if (submitBtn) {
        submitBtn.disabled = false;
        submitBtn.innerHTML = '<i class="fa fa-save"></i> Update Relasi';
    }
    
    const modal = new bootstrap.Modal(document.getElementById('relasiModal'));
    modal.show();
}

// Confirm delete
function confirmDelete(id, produkNama, supplierNama) {
    const confirmItemName = document.getElementById('confirmItemName');
    const confirmDeleteBtn = document.getElementById('confirmDeleteBtn');
    
    if (confirmItemName) confirmItemName.textContent = `${produkNama} - ${supplierNama}`;
    if (confirmDeleteBtn) confirmDeleteBtn.href = '?hapus=' + id;
    
    const modal = document.getElementById('confirmModal');
    if (modal && typeof bootstrap !== 'undefined') {
        new bootstrap.Modal(modal).show();
    }
}

// Auto-hide alerts after 5 seconds
document.addEventListener('DOMContentLoaded', function() {
    const alerts = document.querySelectorAll('.alert-dismissible');
    alerts.forEach(function(alert) {
        setTimeout(function() {
            if (alert && alert.parentNode) {
                const bsAlert = new bootstrap.Alert(alert);
                bsAlert.close();
            }
        }, 5000);
    });
    
    // Initialize Select2
    $('.select2').select2({ 
        dropdownParent: $('#relasiModal'), 
        width: '100%',
        placeholder: function() {
            return $(this).data('placeholder');
        }
    });
    
    // Real-time validation
    const inputs = ['produkSelect', 'supplierSelect', 'hargaInput'];
    inputs.forEach(function(inputId) {
        const input = document.getElementById(inputId);
        if (input) {
            input.addEventListener('change', function() {
                if (this.classList.contains('is-invalid')) {
                    this.classList.remove('is-invalid');
                    const errorElement = document.getElementById(inputId.replace('Select', '').replace('Input', '') + 'Error');
                    if (errorElement) errorElement.textContent = '';
                }
            });
        }
    });
    
    // Format number input
    const hargaInput = document.getElementById('hargaInput');
    if (hargaInput) {
        hargaInput.addEventListener('input', function() {
            // Remove non-numeric characters except decimal point
            this.value = this.value.replace(/[^0-9.]/g, '');
            
            // Ensure only one decimal point
            const parts = this.value.split('.');
            if (parts.length > 2) {
                this.value = parts[0] + '.' + parts.slice(1).join('');
            }
            
            // Remove validation error on input
            if (this.classList.contains('is-invalid')) {
                this.classList.remove('is-invalid');
                const errorElement = document.getElementById('hargaError');
                if (errorElement) errorElement.textContent = '';
            }
        });
    }
});

// Keyboard shortcuts
document.addEventListener('keydown', function(e) {
    // Ctrl+Enter untuk submit form
    if (e.key === 'Enter' && e.ctrlKey) {
        const activeModal = document.querySelector('.modal.show');
        if (activeModal) {
            const submitBtn = activeModal.querySelector('button[type="submit"]');
            if (submitBtn && !submitBtn.disabled) {
                submitBtn.click();
            }
        }
    }
    
    // Ctrl+N untuk tambah baru
    if (e.key === 'n' && e.ctrlKey && !e.shiftKey && !e.altKey) {
        e.preventDefault();
        openModal();
    }
    
    // F5 untuk refresh
    if (e.key === 'F5') {
        e.preventDefault();
        location.reload();
    }
});

// Modal event listeners
document.getElementById('relasiModal').addEventListener('shown.bs.modal', function () {
    // Focus on first select when modal opens
    const produkSelect = document.getElementById('produkSelect');
    if (produkSelect) {
        setTimeout(() => {
            $('#produkSelect').select2('open');
        }, 100);
    }
});

// Reset form when modal is hidden
document.getElementById('relasiModal').addEventListener('hidden.bs.modal', function () {
    const form = this.querySelector('form');
    if (form) {
        // Reset form validation
        ['produkSelect', 'supplierSelect', 'hargaInput'].forEach(fieldId => {
            const field = document.getElementById(fieldId);
            if (field) field.classList.remove('is-invalid');
        });
        
        // Reset submit button
        const submitBtn = document.getElementById('submitBtn');
        if (submitBtn) {
            submitBtn.disabled = false;
            submitBtn.innerHTML = '<i class="fa fa-save"></i> Simpan Relasi';
        }
    }
});

// Initialize tooltips if needed
$(function () {
    $('[data-bs-toggle="tooltip"]').tooltip();
});
</script>

</body>
</html>