<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

include 'config.php';
include 'session.php';

// Check if form is submitted
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $_SESSION['error_message'] = "Method tidak diizinkan";
    header('Location: master_customer.php');
    exit;
}

try {
    // Get form data with validation
    $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
    $kode = trim($_POST['kode'] ?? '');
    $nama = trim($_POST['nama'] ?? '');
    $nama_toko = trim($_POST['nama_toko'] ?? '');
    $merk_toko = trim($_POST['merk_toko'] ?? '');
    $kota = trim($_POST['kota'] ?? '');
    $no_hp = trim($_POST['no_hp'] ?? '');
    $jenis = trim($_POST['jenis'] ?? '');
    $notes = trim($_POST['notes'] ?? '');
    $aktif = isset($_POST['aktif']) ? (int)$_POST['aktif'] : 1;
    $produk_ids = $_POST['produk_ids'] ?? [];
    $query_string = $_POST['query_string'] ?? '';

    // Validation
    $errors = [];
    
    if (empty($nama)) {
        $errors[] = "Nama person tidak boleh kosong";
    }
    
    if (empty($kota)) {
        $errors[] = "Kota tidak boleh kosong";
    }
    
    if (empty($no_hp)) {
        $errors[] = "No. HP tidak boleh kosong";
    }
    
    if (empty($jenis)) {
        $errors[] = "Jenis customer harus dipilih";
    }
    
    // Validate jenis options
    $valid_jenis = ['Sembako', 'Bangunan', 'Sembako dan Bangunan'];
    if (!in_array($jenis, $valid_jenis)) {
        $errors[] = "Jenis customer tidak valid";
    }
    
    // Validate length
    if (strlen($nama) > 100) {
        $errors[] = "Nama person maksimal 100 karakter";
    }
    
    if (strlen($nama_toko) > 100) {
        $errors[] = "Nama toko maksimal 100 karakter";
    }
    
    if (strlen($merk_toko) > 100) {
        $errors[] = "Merk toko maksimal 100 karakter";
    }
    
    if (strlen($kota) > 50) {
        $errors[] = "Kota maksimal 50 karakter";
    }
    
    if (strlen($no_hp) > 20) {
        $errors[] = "No. HP maksimal 20 karakter";
    }
    
    if (strlen($notes) > 500) {
        $errors[] = "Catatan maksimal 500 karakter";
    }

    // If there are validation errors
    if (!empty($errors)) {
        $_SESSION['error_message'] = implode('<br>', $errors);
        $redirect_url = 'master_customer.php';
        if (!empty($query_string)) {
            $redirect_url .= '?' . $query_string;
        }
        header('Location: ' . $redirect_url);
        exit;
    }

    // Generate kode if not provided (for new customer)
    if (empty($kode) && $id == 0) {
        $kode_prefix = strtoupper(substr($kota, 0, 3));
        $query_kode = "SELECT MAX(CAST(SUBSTRING(kode, 4) AS UNSIGNED)) as max_num 
                       FROM customer 
                       WHERE kode LIKE '" . mysqli_real_escape_string($conn, $kode_prefix) . "%'";
        $result_kode = mysqli_query($conn, $query_kode);
        
        if ($result_kode && $row_kode = mysqli_fetch_assoc($result_kode)) {
            $next_num = ($row_kode['max_num'] ?? 0) + 1;
            $kode = $kode_prefix . str_pad($next_num, 3, '0', STR_PAD_LEFT);
        } else {
            $kode = $kode_prefix . '001';
        }
    }

    // Start transaction
    mysqli_begin_transaction($conn);

    if ($id > 0) {
        // Update existing customer
        
        // Check if customer exists
        $check_query = "SELECT id FROM customer WHERE id = ?";
        $check_stmt = mysqli_prepare($conn, $check_query);
        mysqli_stmt_bind_param($check_stmt, "i", $id);
        mysqli_stmt_execute($check_stmt);
        $check_result = mysqli_stmt_get_result($check_stmt);
        
        if (mysqli_num_rows($check_result) == 0) {
            throw new Exception("Customer dengan ID $id tidak ditemukan");
        }
        mysqli_stmt_close($check_stmt);
        
        // Check if column exists (for backward compatibility)
        $columns_query = "SHOW COLUMNS FROM customer LIKE 'nama_toko'";
        $columns_result = mysqli_query($conn, $columns_query);
        $has_new_columns = mysqli_num_rows($columns_result) > 0;
        
        if ($has_new_columns) {
            // New structure with separate fields
            $query = "UPDATE customer SET 
                        kode = ?, 
                        nama = ?, 
                        nama_toko = ?, 
                        merk_toko = ?, 
                        kota = ?, 
                        no_hp = ?, 
                        jenis = ?, 
                        notes = ?, 
                        aktif = ?
                      WHERE id = ?";
            
            $stmt = mysqli_prepare($conn, $query);
            if (!$stmt) {
                throw new Exception("Prepare statement failed: " . mysqli_error($conn));
            }
            
            // Convert empty strings to NULL for optional fields
            $nama_toko_value = !empty($nama_toko) ? $nama_toko : null;
            $merk_toko_value = !empty($merk_toko) ? $merk_toko : null;
            $notes_value = !empty($notes) ? $notes : null;
            
            mysqli_stmt_bind_param($stmt, "ssssssssii", 
                $kode, $nama, $nama_toko_value, $merk_toko_value, 
                $kota, $no_hp, $jenis, $notes_value, $aktif, $id);
        } else {
            // Old structure - combine names
            $combined_nama = $nama;
            if (!empty($nama_toko)) {
                $combined_nama .= " - " . $nama_toko;
            }
            if (!empty($merk_toko)) {
                $combined_nama .= " - " . $merk_toko;
            }
            
            $query = "UPDATE customer SET 
                        kode = ?, 
                        nama = ?, 
                        kota = ?, 
                        no_hp = ?, 
                        jenis = ?, 
                        notes = ?, 
                        aktif = ?
                      WHERE id = ?";
            
            $stmt = mysqli_prepare($conn, $query);
            if (!$stmt) {
                throw new Exception("Prepare statement failed: " . mysqli_error($conn));
            }
            
            $notes_value = !empty($notes) ? $notes : null;
            
            mysqli_stmt_bind_param($stmt, "ssssssii", 
                $kode, $combined_nama, $kota, $no_hp, $jenis, $notes_value, $aktif, $id);
        }
        
        if (!mysqli_stmt_execute($stmt)) {
            throw new Exception("Execute failed: " . mysqli_stmt_error($stmt));
        }
        
        mysqli_stmt_close($stmt);
        
        // Delete existing product relationships
        $delete_produk = "DELETE FROM customer_produk WHERE customer_id = ?";
        $stmt_delete = mysqli_prepare($conn, $delete_produk);
        if (!$stmt_delete) {
            throw new Exception("Prepare delete statement failed: " . mysqli_error($conn));
        }
        
        mysqli_stmt_bind_param($stmt_delete, "i", $id);
        if (!mysqli_stmt_execute($stmt_delete)) {
            throw new Exception("Delete products failed: " . mysqli_stmt_error($stmt_delete));
        }
        mysqli_stmt_close($stmt_delete);
        
        $customer_id = $id;
        $action = "diperbarui";
        
    } else {
        // Insert new customer
        
        // Check if kode already exists
        $check_kode = "SELECT id FROM customer WHERE kode = ?";
        $stmt_check = mysqli_prepare($conn, $check_kode);
        if (!$stmt_check) {
            throw new Exception("Prepare check statement failed: " . mysqli_error($conn));
        }
        
        mysqli_stmt_bind_param($stmt_check, "s", $kode);
        mysqli_stmt_execute($stmt_check);
        $result_check = mysqli_stmt_get_result($stmt_check);
        
        if (mysqli_num_rows($result_check) > 0) {
            throw new Exception("Kode customer '$kode' sudah digunakan");
        }
        mysqli_stmt_close($stmt_check);
        
        // Check if column exists (for backward compatibility)
        $columns_query = "SHOW COLUMNS FROM customer LIKE 'nama_toko'";
        $columns_result = mysqli_query($conn, $columns_query);
        $has_new_columns = mysqli_num_rows($columns_result) > 0;
        
        if ($has_new_columns) {
            // New structure with separate fields
            $query = "INSERT INTO customer 
                        (kode, nama, nama_toko, merk_toko, kota, no_hp, jenis, notes, aktif) 
                      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
            
            $stmt = mysqli_prepare($conn, $query);
            if (!$stmt) {
                throw new Exception("Prepare statement failed: " . mysqli_error($conn));
            }
            
            // Convert empty strings to NULL for optional fields
            $nama_toko_value = !empty($nama_toko) ? $nama_toko : null;
            $merk_toko_value = !empty($merk_toko) ? $merk_toko : null;
            $notes_value = !empty($notes) ? $notes : null;
            
            mysqli_stmt_bind_param($stmt, "ssssssssi", 
                $kode, $nama, $nama_toko_value, $merk_toko_value, 
                $kota, $no_hp, $jenis, $notes_value, $aktif);
        } else {
            // Old structure - combine names
            $combined_nama = $nama;
            if (!empty($nama_toko)) {
                $combined_nama .= " - " . $nama_toko;
            }
            if (!empty($merk_toko)) {
                $combined_nama .= " - " . $merk_toko;
            }
            
            $query = "INSERT INTO customer 
                        (kode, nama, kota, no_hp, jenis, notes, aktif) 
                      VALUES (?, ?, ?, ?, ?, ?, ?)";
            
            $stmt = mysqli_prepare($conn, $query);
            if (!$stmt) {
                throw new Exception("Prepare statement failed: " . mysqli_error($conn));
            }
            
            $notes_value = !empty($notes) ? $notes : null;
            
            mysqli_stmt_bind_param($stmt, "ssssssi", 
                $kode, $combined_nama, $kota, $no_hp, $jenis, $notes_value, $aktif);
        }
        
        if (!mysqli_stmt_execute($stmt)) {
            throw new Exception("Execute failed: " . mysqli_stmt_error($stmt));
        }
        
        $customer_id = mysqli_insert_id($conn);
        mysqli_stmt_close($stmt);
        
        $action = "ditambahkan";
    }

    // Insert product relationships (only if table exists)
    if (!empty($produk_ids) && is_array($produk_ids)) {
        // Check if customer_produk table exists
        $table_check = "SHOW TABLES LIKE 'customer_produk'";
        $table_result = mysqli_query($conn, $table_check);
        
        if (mysqli_num_rows($table_result) > 0) {
            $insert_produk = "INSERT INTO customer_produk (customer_id, produk_id) VALUES (?, ?)";
            $stmt_produk = mysqli_prepare($conn, $insert_produk);
            if (!$stmt_produk) {
                throw new Exception("Prepare product statement failed: " . mysqli_error($conn));
            }
            
            foreach ($produk_ids as $produk_id) {
                $produk_id = (int)$produk_id;
                if ($produk_id > 0) {
                    mysqli_stmt_bind_param($stmt_produk, "ii", $customer_id, $produk_id);
                    if (!mysqli_stmt_execute($stmt_produk)) {
                        throw new Exception("Insert product failed: " . mysqli_stmt_error($stmt_produk));
                    }
                }
            }
            mysqli_stmt_close($stmt_produk);
        }
    }

    // Commit transaction
    mysqli_commit($conn);
    
    // Success message
    $display_name = $nama;
    if (!empty($nama_toko)) {
        $display_name .= " (" . $nama_toko . ")";
    }
    
    $_SESSION['success_message'] = "Customer '$display_name' berhasil $action dengan kode: $kode";
    
    // Redirect back to master customer
    $redirect_url = 'master_customer.php';
    if (!empty($query_string)) {
        $redirect_url .= '?' . $query_string;
    }
    header('Location: ' . $redirect_url);
    exit;

} catch (Exception $e) {
    // Rollback transaction on error
    if (isset($conn)) {
        mysqli_rollback($conn);
    }
    
    $_SESSION['error_message'] = "Error menyimpan data customer: " . $e->getMessage();
    
    $redirect_url = 'master_customer.php';
    if (!empty($_POST['query_string'])) {
        $redirect_url .= '?' . $_POST['query_string'];
    }
    header('Location: ' . $redirect_url);
    exit;
}
?>